/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        $Id: pgpKeyPriv.h,v 1.110.2.5 2001/05/15 20:51:36 ajivsov Exp $
____________________________________________________________________________*/

#ifndef Included_pgpKeyPriv_h
#define Included_pgpKeyPriv_h

#include <stdio.h>

#include "pgpPFLPriv.h"
#include "pgpBinaryTree.h"
#include "pgpEventPriv.h"
#include "pgpFileRef.h"
#include "pgpTypes.h"
#include "pgpUsuals.h"
#include "pgpErrors.h"
#include "pgpMem.h"
#include "pgpMemPool.h"
#include "pgpKeyFilterPriv.h"
#include "pgpPublicKey.h"
#include "pgpSDKPriv.h"


/************************** PGPKeyDB definitions * **************************/

#include "pgpRPCMsg.h"

/* Internal keyring object type definitions */
#define RINGTYPE_KEY	1
#define RINGTYPE_SUBKEY	2
#define RINGTYPE_USERID	3
#define RINGTYPE_SIG	4
#define RINGTYPE_CRL	5
#define RINGTYPE_UNK	6	/* Object of unknown type */
#define RINGTYPE_MAX	6

#define OBJISTOPKEY(obj) (pgpObjectType(obj)==RINGTYPE_KEY)
#define OBJISSUBKEY(obj) (pgpObjectType(obj)==RINGTYPE_SUBKEY)
#define OBJISUSERID(obj) (pgpObjectType(obj)==RINGTYPE_USERID)
#define OBJISSIG(obj)	 (pgpObjectType(obj)==RINGTYPE_SIG)
#define OBJISCRL(obj)	 (pgpObjectType(obj)==RINGTYPE_CRL)
#define OBJISKEY(obj)	 (OBJISTOPKEY(obj) || OBJISSUBKEY(obj))

/* Maximum size of a keyring object we try to read */
#define PGPOBJ_MAXSIZE		150000

/* CRL type values */
#define PGPCRLTYPE_X509			1
#define PGPCRLTYPE_X509DPOINT	2

/* Signature subpacket types.  Many of these are not supported yet. */
enum sigsubbyte {
	/* Signature specific properties */
	SIGSUB_VERSION			= 1,
	SIGSUB_CREATION,
	SIGSUB_EXPIRATION,
	SIGSUB_EXPORTABLE,
	SIGSUB_TRUST,
	SIGSUB_REGEXP,
	SIGSUB_REVOCABLE,
	/* Properties of key being self signed */
	SIGSUB_KEY_CAPABILITIES = 8,
	SIGSUB_KEY_EXPIRATION,
	SIGSUB_KEY_ADDITIONAL_RECIPIENT_REQUEST,
	SIGSUB_PREFERRED_ENCRYPTION_ALGS,
	SIGSUB_KEY_REVOCATION_KEY,
	/* Hints to find signer */
	SIGSUB_KEYID			= 16,
	SIGSUB_USERID,
	SIGSUB_URL,
	SIGSUB_FINGER,
	/* Miscellaneous packets */
	SIGSUB_NOTATION			= 20,
	SIGSUB_PREFERRED_HASH_ALGS,
	SIGSUB_PREFERRED_COMPRESSION_ALGS,
	SIGSUB_KEYSERVER_PREFERENCES,
	SIGSUB_PREFERRED_KEYSERVER,
	SIGSUB_PRIMARY_USERID,
	SIGSUB_POLICY_URL,
	SIGSUB_KEYFLAGS,
	SIGSUB_SIGNER_USERID	= 28,
	SIGSUB_REVOCATION_REASON,
	/* 100-110 are for private use */
	/* Reserve 100 for internal Network Associates use */
	SIGSUB_NAI				= 100,
	/* Used internally for unrecognized packet types */
	SIGSUB_UNRECOGNIZED		= 255	/* crit bit is ignored here */
};

/* Critical bit means we must handle this subpacket */
#define SIGSUBF_CRITICAL	0x80

/* Flag bits within SIGSUB_KEYFLAGS */
#define SIGSUBF_KEYFLAG0_USAGE_CERTIFY			0x01
#define SIGSUBF_KEYFLAG0_USAGE_SIGN				0x02
#define SIGSUBF_KEYFLAG0_USAGE_ENCRYPT_COMM		0x04
#define SIGSUBF_KEYFLAG0_USAGE_ENCRYPT_STORAGE	0x08
/* Meaningful only on self signature */
#define SIGSUBF_KEYFLAG0_PRIVATE_SPLIT			0x10
#define SIGSUBF_KEYFLAG0_PRIVATE_SHARED			0x80

/* Flag bits within SIGSUB_KEYSERVER_PREFERENCES */
#define SIGSUBF_KEYSERVER0_

/* Signature sub-subpacket types, within SIGSUB_NAI packets */
enum sigsubsubbyte {
	SIGSUBSUB_X509			= 1
};

/* Version of X509 translation code */
#define SIGSUBSUB_X509_VERSION_HI	0x01
#define SIGSUBSUB_X509_VERSION_LO	0x04

#define pgpKeyUse(k)					pgpKeyUseInternal(k,FALSE,FALSE)
#define pgpKeyUnexpiredUse(k)			pgpKeyUseInternal(k,TRUE,FALSE)
#define pgpKeyUseRevokedOK(k)			pgpKeyUseInternal(k,FALSE,TRUE)

/* CRL type values */
#define PGPCRLTYPE_X509			1
#define PGPCRLTYPE_X509DPOINT	2




/************************ PGPKeyDB Data Structures **************************/


/* This is a path or a path segment.  next->src may be unequal to dest. */
typedef struct Path {
	struct Path         *next;
	PGPKeyDBObj         *src,
	                    *dest;
	double               confidence;
	DEBUG_STRUCT_CONSTRUCTOR( Path )
} Path;

/* This is a list of paths.  Some segments may be on multiple paths. */
typedef struct PathList {
	struct PathList     *next;
	Path                *path;
	Path               **ptail;
	double               confidence;
	DEBUG_STRUCT_CONSTRUCTOR( PathList )
} PathList, *pPathList;


struct PGPKeyDB
{
#define kPGPKeyDBMagic		0x4B594442	/* 'KYDB' */
	PGPUInt32			fixedMagic;
	
	struct PGPKeyDB *next;			/* Next keydb in list */
	PGPFile *		pubFile;		/* PGPFile for public keys */
	PGPFile *		privFile;		/* PGPFile for private keys */
	PFLFileSpecRef	pubFileRef;		/* Public keyring filename */
	PFLFileSpecRef	privFileRef;	/* Private keyring filename */
	FILE *			pubStdFile;		/* Pub keyring FILE (for locking only) */
	FILE *			privStdFile;	/* Priv keyring FILE (for locking only) */
	PGPBoolean		bmutable;		/* True if a writeable database */
	PGPBoolean		bdirty;			/* True if mutableSet has been changed */
	PGPFlags		openFlags;		/* Flags used to open keyring */

	PGPInt32			refCount;
	PGPContextRef		context;
	PGPKeySetRef		rootSet;

/* Linked list of all sets, used for cleanup */
	PGPKeySetRef		firstSetInDB;

/* Allocation of new objects */
	MemPool				objPool;	/* For key objects */
	MemPool				structPool;	/* For key sets and iters and such */
	PGPKeyDBObj *		oldobjs[2];	/* Cache to speed up adding keys */
	PGPKeyDBObj *		newobjs[2];

/* Pointer to actual key data */
	PGPKeyDBObj *		firstKeyInDB;
	PGPKeyDBObj *		lastKeyInDB;
	PGPKeyDBObj *		treeKey;		/* keyid-sorted tree */
	long				numKeys;


	/* Values used for trust calculations */
    MemPool				pathpool;
    Path *				paths;
	PathList *			pathlists;
	MemPool				regexps;

	/* In case of error, the following is set */
	PGPError			err;

/* Front end only */
	PGPUInt32			id;			/* Corresponding backend keydb */
	PGPBinaryTreeRef	idToObj;	/* Translate backend ids to keydbobjs */

/* Back end only */
	struct PGPNotification *notifies;	/* Pending notifications */

	DEBUG_STRUCT_CONSTRUCTOR( PGPKeyDB )
};

/************************ PGPKeySet Data Structures *************************/


/*
 * These data structures do not span frontend/backend boundaries.  Each one
 * is either entirely in front end or entirely in back end.
 */

/*
 * A KeySet is the result of applying a filter (or series of filters) to
 * the KeyDB.  It does not have an explicit list of the objects in the
 * set, but rather those are determined when we actually use it, either
 * when we sort it or iterate over it.
 */
struct PGPKeySet
{
#define kPGPKeySetMagic	0x4B534554	/* 'KSET' */
	PGPUInt32			magic;
	void *				priv;			/* Private data, used for filtering */
	PGPInt32			refCount;
	PGPKeyDBRef			keyDB;
	PGPKeySetRef		prevSetInDB;
	PGPKeySetRef		nextSetInDB;
	PGPKeyListRef		firstListInSet;
	PGPBoolean			filterSubObjects;

	PGPBoolean			(*isMember)	(PGPKeySetRef set, PGPKeyDBObjRef key);
	PGPBoolean			(*isEmpty)	(PGPKeySetRef set);
	PGPError			(*makeUnion)(PGPKeySetRef set1, PGPKeySetRef set2,
									 PGPKeySetRef *newSet);
	void				(*destroy)	(PGPKeySetRef set);

	DEBUG_STRUCT_CONSTRUCTOR( PGPKeySet )
};


/*
 * A KeyList is a sorted version of a KeySet.  One type of "sort" is
 * unordered.  If it is an ordered sort, we identify which keys are in
 * the KeyList at the time it is created, and put them in the keys array.
 * For an unordered sort, we don't do this, and we leave keys as null and
 * keyCount as zero.
 */
struct PGPKeyList
{
#define kPGPKeyListMagic	0x4B4C5354	/* 'KLST' */
	PGPUInt32			magic;
	PGPInt32			refCount;
	PGPKeySetRef		keySet;
	PGPKeyOrdering		order;
	PGPByte				reverse;		/* 0 or 1 */

	PGPKeyListRef		prevListInSet;
	PGPKeyListRef		nextListInSet;
	PGPKeyIterRef		firstIterInList;

	long				keyCount;		/* size of keys array */
	PGPKeyDBObjRef*		keys;

	DEBUG_STRUCT_CONSTRUCTOR( PGPKeyList )
} ;


/* A KeyIter is used to iterate over a KeyList. */
struct PGPKeyIter
{
#define kPGPKeyIterMagic	0x4B495452	/* 'KITR' */
	PGPUInt32			magic;
	
	PGPBoolean			isKeyList;		/* based on keylist vs keyset */
	PGPKeyListRef		keyList;		/* parent keylist */
	PGPKeySetRef		keySet;			/* parent keyset */

	PGPKeyIterRef		prevIterInList;	/* next iter in keylist */
	PGPKeyIterRef		nextIterInList;	/* next iter in keylist */

	long				keyIndex;		/* current key index in keylist */
	PGPKeyDBObjRef		currentObject;	/* ptr to current obj */
	PGPInt8				level;			/* level of obj in hierarchy */
	PGPBoolean			atEndOfLevel;	/* at end of current level? */

	DEBUG_STRUCT_CONSTRUCTOR( PGPKeyIter )
} ;


/*********************** PGPKeyDBObj Data Structures ************************/



/* Generic reference to any kind of keydb object */
struct PGPKeyDBObj {
#define kPGPKeyDBObjMagic		0x4B4F424A	/* 'KOBJ' */
	PGPUInt32		magic;
	PGPKeyDBObj *	next;					/* Next object at this level */
	PGPKeyDBObj *	down;					/* First child of this object */
	PGPKeyDBObj *	up;						/* If not a key, parent obj */
											/* If a key, keydb */
	PGPUserValue	userVal;				/* User settable value */
	union {
		PGPUInt32				id;			/* ID to retrieve info */
		PGPKeyDBObjInfo *		info;		/* Cached info about obj */
	}				 idinfo;
	PGPUInt32		 objflags;
		/* flags includes:
		 *		type of object - key vs sig vs userid vs subkey etc
		 *		deleted flag
		 *		"info vs id present" flag for idinfo field
		 */

	DEBUG_STRUCT_CONSTRUCTOR( PGPKeyDBObj )
};
/* objflags for private use, in addition to object type flags */
#define kPGPKeyDBObjType_CRL			(1UL << 4)
#define kPGPKeyDBObjFlags_Deleted		0x80000000
#define kPGPKeyDBObjFlags_Dummy			0x40000000
/* X509 means a dummy key whose id is the hash of the X.509 subject name */
/* Such keys have their "data" pointer pointing at the cert sig */
#define kPGPKeyDBObjFlags_X509			0x20000000
/* ID present in idinfo field rather than info */
#define kPGPKeyDBObjFlags_ID			0x10000000


typedef struct DataHeader {
	PGPByte const *	data;
	PGPSize			len;
	PGPUInt32		refcount;
} DataHeader;


struct PGPKeyInfo {
	PGPUInt32			id;				/* ID to retrieve info */
	PGPByte				pkalg;			/* PK algorithm */
	PGPUInt16			keybits;		/* Key size in bits */
	PGPByte				keyID[8];		/* Key ID */
	PGPUInt32			creationtime;	/* Time key was created */
	PGPUInt16			validityperiod;/* creation to expiration */
	DataHeader *		data;			/* Raw data of object */

	PGPUInt32		pos;				/* Backing store data */
	PGPUInt32		flags;				/* Extra info about the object */
	PGPByte			fp20n;				/* First byte from fp20n, for x509 */
	PGPKeyDBObj *	util;				/* Used during trust calcs */
	PGPKeyDBObj *	sigsby;				/* Chain of sigs by this key */
	PGPKeyDBObj *	left;				/* Keys with lesser keyids */
	PGPKeyDBObj *	right;				/* Keys with greater keyids */
	PGPByte			signedTrust;		/* Trust from sigs on key */
	PGPByte			trust;				/* User-specified trust on key */
	void *			regexp;				/* reg exp during trust calculation */

	PGPInt32		tokenNum1;			/* token number, starting from 1, if 
	                                       KEYISTOKEN */

	DEBUG_STRUCT_CONSTRUCTOR( PGPKeyInfo )
};
/* Access the key flags field */
#define KEYF_TRUSTTMP		0x80		/* Temporary used during trust calc */
#define KEYF_V3				0x40		/* V2 or V3 key (earlier than V4) */
#define KEYF_VBUG			0x20		/* Sec key V3 should be V2 */
#define KEYF_SEC			0x10		/* Key is a secret key */
#define KEYF_TOKEN			0x08		/* Secret part of key is on a token */
#define KEYF_VALIDATED		0x04		/* Secret part has been validated */

#define KEYISV3(k) (((k)->flags & KEYF_V3) != 0)
#define KEYSETV3(k) ((k)->flags |= KEYF_V3)
#define KEYCLEARV3(k) ((k)->flags &= ~KEYF_V3)
#define KEYHASVERSIONBUG(k) (((k)->flags & KEYF_VBUG) != 0)
#define KEYSETVERSIONBUG(k) ((k)->flags |= KEYF_VBUG)
#define KEYCLEARVERSIONBUG(k) ((k)->flags &= ~KEYF_VBUG)
#define KEYISSEC(k) (((k)->flags & KEYF_SEC) != 0)
#define KEYSETSEC(k) ((k)->flags |= KEYF_SEC)
#define KEYCLEARSEC(k) ((k)->flags &= ~KEYF_SEC)
#define KEYISTOKEN(k) (((k)->flags & KEYF_TOKEN) != 0)
#define KEYSETTOKEN(k) ((k)->flags |= KEYF_TOKEN)
#define KEYCLEARTOKEN(k) ((k)->flags &= ~KEYF_TOKEN)
#define KEYISVALIDATED(k) (((k)->flags & KEYF_VALIDATED) != 0)
#define KEYSETVALIDATED(k) ((k)->flags |= KEYF_VALIDATED)
#define KEYCLEARVALIDATED(k) ((k)->flags &= ~KEYF_VALIDATED)

struct PGPUserIDInfo
{
	PGPUInt32		 id;				/* ID to retrieve info */
	DataHeader *	data;				/* Raw data of object */
	PGPByte			validity;			/* Stored validity */

	PGPUInt32		pos;				/* Backing store data */
	PGPUInt32		flags;				/* Extra info about the object */
	PGPByte			oldvalidity;		/* Old 4-level validity */
	PGPByte			confidence;			/* Stored confidence */
	PGPUInt16		valid;				/* Computed validity */

	DEBUG_STRUCT_CONSTRUCTOR( PGPUserIDnfo )
};

/* Access the name flags field */
#define NAMEF_ATTR			0x80		/* Attribute userid */

#define NAMEISATTR(n) (((n)->flags & NAMEF_ATTR) != 0)
#define NAMESETATTR(n) ((n)->flags |= NAMEF_ATTR)
#define NAMECLEARATTR(n) ((n)->flags &= ~NAMEF_ATTR)



struct PGPSigInfo
{
	PGPUInt32			id;				/* ID to retrieve info */
	DataHeader *		data;			/* Raw data of object */
	PGPByte				type;			/* Sig type (regular, revoke...) */
	PGPByte				hashalg;		/* Hash alg used by sig */
	PGPByte				trustLevel;	/* 0: regular sig, 1: trust, 2: meta */
	PGPByte				trustValue;	/* Used if trustLevel>0 */
	PGPUInt32			tstamp;		/* Creation date */
	PGPUInt32			sigvalidity;	/* Time until expiration */

	PGPUInt32		pos;				/* Backing store data */
	PGPUInt32		flags;				/* Extra info about the object */
	PGPByte			trust;
	PGPByte			version;			/* V3 vs V4 signature */
	PGPKeyDBObj *	by;					/* Key which created sig */
	PGPKeyDBObj *	by2;				/* Alternate signer, for trust calc */
	PGPKeyDBObj *	nextby;				/* Next sig by same key */
	void *			regexp;				/* Reg exp for propagating trust */

	DEBUG_STRUCT_CONSTRUCTOR( PGPSigInfo )
};

/* Access the sig flags field */
#define SIGF_EXPORTABLE		0x80	/* Sig can be exported to others */
#define SIGF_USESREGEXP	 	0x40	/* (Trust) sig qualified with regexp */
#define SIGF_REVOCABLE		0x20	/* Sig can be revoked */
#define SIGF_X509			0x10	/* Sig is an imported X.509 cert */
#define SIGF_PRIMARYUID	 	0x08	/* Sig says it's on primary userid */
#define SIGF_DISTPOINT		0x04	/* Sig has a distribution point */
#define SIGF_NONFIVE		0x02	/* Extralen != 5 */

#define SIGSETEXPORTABLE(s)			(s)->flags |= SIGF_EXPORTABLE
#define SIGSETNONEXPORTABLE(s)		(s)->flags &= ~SIGF_EXPORTABLE
#define SIGISEXPORTABLE(s)			(((s)->flags & SIGF_EXPORTABLE)!=0)
#define SIGSETUSESREGEXP(s)			(s)->flags |= SIGF_USESREGEXP
#define SIGCLEARUSESREGEXP(s)		(s)->flags &= ~SIGF_USESREGEXP
#define SIGUSESREGEXP(s)			(((s)->flags & SIGF_USESREGEXP)!=0)
#define SIGSETREVOCABLE(s)			(s)->flags |= SIGF_REVOCABLE
#define SIGSETNONREVOCABLE(s)		(s)->flags &= ~SIGF_REVOCABLE
#define SIGISREVOCABLE(s)			(((s)->flags & SIGF_REVOCABLE)!=0)
#define SIGSETX509(s)				(s)->flags |= SIGF_X509
#define SIGCLEARX509(s)				(s)->flags &= ~SIGF_X509
#define SIGISX509(s)				(((s)->flags & SIGF_X509)!=0)
#define SIGSETPRIMARYUID(s)			(s)->flags |= SIGF_PRIMARYUID
#define SIGCLEARPRIMARYUID(s)		(s)->flags &= ~SIGF_PRIMARYUID
#define SIGISPRIMARYUID(s)			(((s)->flags & SIGF_PRIMARYUID)!=0)
#define SIGSETDISTPOINT(s)			(s)->flags |= SIGF_DISTPOINT
#define SIGCLEARDISTPOINT(s)		(s)->flags &= ~SIGF_DISTPOINT
#define SIGHASDISTPOINT(s)			(((s)->flags & SIGF_DISTPOINT)!=0)

typedef struct PGPCRLInfo
{
	PGPUInt32			id;				/* ID to retrieve info */
	DataHeader *		data;			/* Raw data of object */
	PGPUInt32			flags;			/* Extra info about the object */
	PGPByte				version;		/* Version of subpacket */
	PGPByte				trust;			/* Trust info */
	PGPUInt32			tstamp;			/* Issuance date of CRL */
	PGPUInt32			tstampnext;		/* Next-issuance date of CRL */
} PGPCRLInfo;

#define CRLF_X509			 0x80	/* Crl is an X509 CRL */
#define CRLF_DPOINT			 0x40	/* Crl has a distribution point */

#define CRLSETX509(c)				(c)->flags |= CRLF_X509
#define CRLCLEARX509(c)				(c)->flags &= ~CRLF_X509
#define CRLISX509(c)				(((c)->flags & CRLF_X509)!=0)
#define CRLSETDPOINT(c)				(c)->flags |= CRLF_DPOINT
#define CRLCLEARDPOINT(c)			(c)->flags &= ~CRLF_DPOINT
#define CRLHASDPOINT(c)				(((c)->flags & CRLF_DPOINT)!=0)



typedef struct PGPUnkInfo
{
	PGPUInt32			id;				/* ID to retrieve info */
	DataHeader *		data;			/* Raw data of object */
	PGPByte				trust;
} PGPUnkInfo;


union PGPKeyDBObjInfo {
	PGPKeyInfo			keyinfo;
	PGPUserIDInfo		useridinfo;
	PGPSigInfo			siginfo;
	PGPCRLInfo			crlinfo;
	PGPUnkInfo			unkinfo;

	DEBUG_STRUCT_CONSTRUCTOR( PGPKeyDBObjInfo )
};




#define CHECKREMOVED(x)		\
if (IsNull(x) || pgpKeyDBObjIsDeleted(x))	\
	return kPGPError_ItemWasDeleted



PGP_BEGIN_C_DECLARATIONS





/* pgpEnumeratedSet.c */
PGPError				pgpNewKeyListSet( PGPKeyDB *db,PGPKeyDBObjRef *keylist,
							PGPUInt32 nkeys, PGPKeySetRef *newSet );


/* pgpKeyBack.c */

PGPError				pgpFetchObjectData_back( PGPContextRef context,
							PGPUInt32 id, PGPByte **bufptr, PGPSize *buflen );
PGPError				pgpGetKeyByKeyID_back( PGPContextRef context,
							PGPUInt32 dbid, PGPKeyID const *keyIDIn,
							PGPBoolean dummyOK, PGPBoolean deletedOK,
							PGPUInt32 *outID );
PGPError				pgpKeyEncrypt_back( PGPContextRef context,
							PGPUInt32 id, PGPByte const *inbuf,
							PGPSize inbuflen, PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPError				pgpKeyDecrypt_back( PGPContextRef context,
							PGPUInt32 id, PGPByte const *passphrase,
							PGPSize pplen, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPByte const *inbuf, PGPSize inbuflen,
							PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPInt32				pgpKeyVerify_back( PGPContextRef context,
							PGPUInt32 id, PGPByte const *inbuf,
							PGPSize inbuflen, PGPHashAlgorithm hashalg,
							PGPByte const *hash, PGPSize hashlen,
							PGPPublicKeyMessageFormat format);
PGPError				pgpKeySign_back( PGPContextRef context, PGPUInt32 id,
							PGPByte const *passphrase, PGPSize pplen,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPHashAlgorithm hashalg,
							PGPByte const *hash, PGPSize hashlen,
							PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPBoolean				pgpSecPassphraseOK_back( PGPContextRef context,
							PGPUInt32 id, PGPByte const *passphrase,
							PGPSize pplen, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal );
PGPError				pgpKeyMaxSizes_back( PGPContextRef context,
							PGPUInt32 id, PGPUInt32 *maxEncryption,
							PGPUInt32 *maxDecryption, PGPUInt32 *maxSignature,
							PGPPublicKeyMessageFormat format );
PGPError				pgpSecProperties_back( PGPContextRef context,
							PGPUInt32 id, PGPBoolean *needsPassphrase,
							PGPBoolean *isSecretShared,
							PGPCipherAlgorithm *lockAlg, PGPUInt32 *lockBits );
PGPError				pgpFetchKeyInfo_back( PGPContextRef context,
							PGPUInt32 id, PGPByte **bufptr, PGPSize *bufsize );
PGPError				pgpOpenKeyDBFile_back(PGPContextRef cdkContext,
							PGPOpenKeyDBFileOptions openFlags,
							PFLFileSpecRef pubFileRef,
							PFLFileSpecRef privFileRef, PGPUInt32 * kdbid,
							PGPUInt32 * numKeys, PGPUInt32 ** keyArray,
							PGPSize * keyArraySize );
PGPError				pgpNewKeyDB_back( PGPContextRef context,
							PGPUInt32 *keydbid );
PGPError				pgpKeyDBArray_back( PGPContextRef cdkContext,
							PGPUInt32 kdbid, PGPUInt32 * numKeys,
							PGPUInt32 ** keyArray, PGPSize * keyArraySize );
PGPError				pgpUpdateKeyDB_back(PGPContextRef context,
							PGPUInt32 kdbid, PGPUInt32 * numNewKeys,
							PGPUInt32 ** newKeyArray,
							PGPSize * newKeyArraySize,
							PGPUInt32 ** changedkeylist,
							PGPSize * changedkeylistsize );
PGPError				pgpKeyDBFlush_back( PGPContextRef context,
							PGPUInt32 id, PGPUInt32 **changedkeylist,
							PGPSize *changedkeylistsize );
void					pgpFreeKeyDB_back( PGPContextRef context,
							PGPUInt32 id );
PGPError				pgpSetKeyEnabled_back( PGPContextRef context,
							PGPUInt32 id, PGPBoolean enable );
PGPError				pgpSetKeyAxiomatic_back( PGPContextRef context,
							PGPUInt32 id, PGPBoolean setAxiomatic,
							PGPBoolean checkPassphrase,
							char const *passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal);
PGPError				pgpPropagateTrust_back( PGPContextRef context,
							PGPUInt32 setid, PGPUInt32 *keylist,
							PGPSize keylistsize, PGPUInt32 altid,
							PGPUInt32 const timenow,
							PGPUInt32 **changedkeylist,
							PGPSize *changedkeylistsize );
PGPError				pgpCheckKeyRingSigs_back( PGPContextRef context,
							PGPUInt32 setid, PGPUInt32 *keylist,
							PGPSize keylistsize, PGPUInt32 altid,
							PGPBoolean checkAll, PGPUInt32 **changedkeylist,
							PGPSize *changedkeylistsize );
PGPError				pgpPrepareToCheckKeyRingSigs_back(
							PGPContextRef context, PGPUInt32 setid,
							PGPUInt32 *keylist, PGPSize keylistsize,
							PGPUInt32 altid, PGPBoolean checkAll,
							PGPUInt32 *nsigs, PGPUInt32 **changedkeylist,
							PGPSize *changedkeylistsize );
PGPError				pgpCheckSig_back( PGPContextRef context,
							PGPUInt32 setid, PGPUInt32 objid,
							PGPUInt32 altid, PGPBoolean checkAll,
							PGPBoolean revocationonly, PGPBoolean *handled,
							PGPBoolean *changed, PGPBoolean *verified );
PGPError				pgpDoGenerateKey_back( PGPContextRef context,
							PGPUInt32 keydbID, PGPUInt32 masterkeyID,
							PGPByte pkalg, PGPUInt32 bits,
							PGPTime creationDate, PGPUInt16 expirationDays,
							char const * name, PGPSize name_len,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean passphraseIsKey,
							char const * masterpass, PGPSize masterpassLength,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPEventHandlerProcPtr progress,
							PGPUserValue userValue,
							PGPBoolean fastgen, PGPBoolean checkentropy,
							PGPBoolean useToken, PGPUInt32 tokenID,
							PGPUInt32 * adkkeylist, PGPSize adkkeylistsize,
							PGPByte adkclass, PGPUInt32 * rakkeylist,
							PGPSize rakkeylistsize, PGPByte rakclass,
							PGPCipherAlgorithm const * prefalg,
							PGPSize prefalgLength, PGPByte const * prefkeyserv,
							PGPSize prefkeyservLength, PGPUInt32 keyflags,
							PGPBoolean fkeyflags, PGPUInt32 keyservprefs,
							PGPBoolean fkeyservprefs, PGPUInt32 **newobjs,
							PGPSize *newobjsLength, PGPUInt32 * newKey);
PGPError				pgpCountTokens_back (PGPContextRef context,
							PGPUInt32 *numTokens );
PGPError				pgpGetTokenInfo_back (PGPContextRef context,
							PGPUInt32 tokNumber, PGPTokenInfo *tokenInfo );
PGPError				pgpDeleteKeyOnToken_back (PGPContextRef context, 
							PGPUInt32 keydbID, const PGPKeyID *keyID, 
							PGPUInt32 tokNumber, 
							PGPByte const *pin, PGPSize pinLen );
PGPError				pgpWipeToken_back (PGPContextRef context,
							PGPUInt32 tokNumber, PGPByte const *passphrase,
							PGPSize passphraseLength );
PGPError				pgpTokenPassphraseIsValid_back (PGPContextRef context,
							PGPUInt32 tokNumber, PGPByte const *passphrase,
							PGPSize passphraseLength );
PGPError                pgpSetPKCS11DrvFile_back( PGPByte *module );
PGPError				pgpCopyKeyToToken_back (PGPContextRef context,
							PGPUInt32 id, PGPUInt32 tokNumber,
							PGPBoolean isMaster,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, char const * PIN,
							PGPSize PINlength, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal);
PGPError                pgpTokenImportX509_back (PGPContextRef	context,
	                        const PGPByte *keyID, 
                            const PGPByte *userID, PGPSize userIDlen, 
                            const PGPByte *x509, PGPSize x509len,
                            const PGPByte *password, PGPSize passwordLength );
PGPError                pgpTokenPutKeyContainer_back( 
                            PGPContextRef context, const PGPByte *keyID, 
                            const PGPByte *password, PGPSize passwordLen, 
                            const PGPByte *cont, PGPSize contSize );
PGPError				pgpTokenGetKeyContainer_back( PGPContextRef context,
							const PGPByte *keyID,
							const PGPByte *password, PGPSize passwordLen, 
							PGPByte	**contOut, PGPSize *contOutSize );
PGPError				pgpAddUserID_back (PGPContextRef context,
							PGPUInt32 id, PGPBoolean isAttribute,
							PGPAttributeType attributeType,
							char const * userIDData, PGPSize userIDLength,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPUInt32 **newobjs,
							PGPSize *newobjsLength );
PGPError				 pgpCertifyUserID_back(PGPContextRef context,
							PGPUInt32 useridid, PGPUInt32 certifying_keyid,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPBoolean exportable,
							PGPTime creationDate, PGPUInt32 expiration,
							PGPByte trustDepth, PGPByte trustValue,
							char const * sRegExp, PGPSize sRegExpLength,
							PGPUInt32 **newobjs, PGPSize *newobjsLength );
PGPError				pgpCertifyPrimaryUserID_back (PGPContextRef ctxt,
							PGPUInt32 useridid, char *passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 **newobjs, PGPSize *newobjsLength);
PGPError				pgpRevokeSig_back (PGPContextRef context,
							PGPUInt32 sigid, char const * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 **newobjs, PGPSize *newobjsLength);
PGPError				pgpRevokeKey_back (PGPContextRef context,
							PGPUInt32 key_id, char const * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 **newobjs, PGPSize *newobjsLength);
PGPError				pgpDoChangePassphrase_back (PGPContextRef context,
							PGPUInt32 keydbid, PGPUInt32 key_id,
							PGPUInt32 masterkey_id, const char *oldphrase,
							PGPSize oldphraseLength, const char *newphrase,
							PGPSize newphraseLength,
							PGPBoolean newPassphraseIsKey,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal);
PGPError				pgpPassphraseIsValid_back( PGPContextRef context,
							PGPUInt32 key_id, const char * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPBoolean * isValid);
PGPError				pgpPurgePassphraseCache_back( PGPContextRef context );
PGPError				pgpSetKeyTrust_back( PGPContextRef context,
							PGPUInt32 key_id, PGPUInt32 trust );
PGPError				pgpGetPasskeyBuffer_back( PGPContextRef context,
							PGPUInt32 key_id, char const * passphrase,
							PGPSize passphraseLength, PGPByte **passkeyBuffer,
							PGPSize *passkeyBufferLength );
PGPError				pgpAddKeyOptions_back (PGPContextRef context,
							PGPUInt32 key_id, char * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 *raklist, PGPSize raklistsize,
							PGPUInt32 rakclass, PGPUInt32 **newobjs,
							PGPSize *newobjsLength);
PGPError				pgpUpdateKeyOptions_back (PGPContextRef context,
							PGPUInt32 key_id, char * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPCipherAlgorithm *prefalg,
							PGPSize prefalgLength, PGPByte *prefkeyserv,
							PGPSize prefkeyservLength, PGPUInt32 keyflags,
							PGPBoolean fkeyflags, PGPUInt32 keyservprefs,
							PGPBoolean fkeyservprefs, PGPUInt32 **newobjs,
							PGPSize *newobjsLength );
PGPError				pgpKeyDBAddObject_back (PGPContextRef context,
							PGPUInt32 kdb_id, PGPUInt32 obj_id,
							PGPUInt32 **newobjs, PGPSize *newobjsLength,
							PGPUInt32 *pnewobj);
PGPError				pgpKeyDBRemoveObject_back (PGPContextRef context,
							PGPUInt32 kdb_id, PGPUInt32 obj_id);
PGPError				pgpImportKeyBinary_back(PGPContextRef cdkContext,
							PGPByte *buffer, PGPSize length,
							PGPUInt32 *kdbid, PGPUInt32 *numKeys,
							PGPUInt32 **keyArray, PGPSize *keyArraySize );
PGPError				pgpGlobalRandomPoolAddState_back(
							PGPBoolean addKeyState, PGPInt32 keyEvent,
							PGPBoolean addMouseState,
							PGPBoolean addSystemState );
PGPError				pgpGlobalRandomPoolGetInfo_back( PGPUInt32 *entropy,
							PGPUInt32 *size, PGPUInt32 *minEntropy,
							PGPBoolean *hasMinEntropy,
							PGPBoolean *hasIntelRNG );
PGPError				pgpRandomGetBytesEntropy_back( PGPContextRef context,
							PGPUInt32 requestBytes, PGPUInt32 entropyBits,
							PGPByte **randBuf, PGPSize *randBufLen );
PGPError				pgpRandomAddBytes_back( PGPContextRef context,
							PGPByte const *buf, PGPSize buflen );
PGPError				pgpRandomStir_back( PGPContextRef context );
PGPError				pgpCopyKeys_back( PGPContextRef context,
							PGPUInt32 srcid, PGPUInt32 dstid,
							PGPUInt32 *keylist, PGPSize keylistsize,
							PGPBoolean neednewkeylist, PGPUInt32 **newkeylist,
							PGPSize *newkeylistsize );
PGPError				pgpSetRandSeedFile_back( PFLFileSpecRef randFile );
PGPError				pgpCreateKeypair_back( PGPContextRef context,
							PGPUInt32 keydbID, PGPByte * seckeyBuf,
							PGPSize seckeySize, PGPByte * keyspecBuf,
							PGPSize keyspecSize, char const * name,
							PGPSize name_len,
							PGPUInt32 * adklist, PGPSize adklistsize,
							PGPByte adkclass,
							PGPUInt32 * raklist, PGPSize raklistsize,
							PGPByte rakclass,
							char const * passphrase,
							PGPSize passphraseLength, PGPBoolean
							passphraseIsKey, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPUInt32 **
							newobjs, PGPSize * newobjslength,
							PGPUInt32 * newkeyid );
PGPError				pgpCreateSubkeypair_back( PGPContextRef context,
							PGPUInt32 masterkeyID, PGPByte * seckeyBuf,
							PGPSize seckeySize, PGPByte * keyspecBuf,
							PGPSize keyspecSize, char const * passphrase,
							PGPSize passphraseLength,
							PGPBoolean passphraseIsKey, PGPUInt32
							cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 ** newobjs, PGPSize *
							newobjslength, PGPUInt32 * newkeyid );



/* pgpRPCKeyBack.c */
#if PGP_MACINTOSH && PGPSDK_BACKEND
#pragma export on
#endif
PGPError				pgpFetchObjectData_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPByte **bufptr, PGPSize *buflen );
PGPError				pgpGetKeyByKeyID_backRPC( PGPContextRef context,
							PGPUInt32 dbid, PGPKeyID const *keyIDIn,
							PGPBoolean dummyOK, PGPBoolean deletedOK,
							PGPUInt32 *outID );
PGPError				pgpKeyEncrypt_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPByte const *inbuf,
							PGPSize inbuflen, PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPError				pgpKeyDecrypt_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPByte const *passphrase,
							PGPSize pplen, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPByte const *inbuf, PGPSize inbuflen,
							PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPInt32				pgpKeyVerify_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPByte const *inbuf,
							PGPSize inbuflen, PGPHashAlgorithm hashalg,
							PGPByte const *hash, PGPSize hashlen,
							PGPPublicKeyMessageFormat format);
PGPError				pgpKeySign_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPByte const *passphrase,
							PGPSize pplen, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPHashAlgorithm hashalg, PGPByte const *hash,
							PGPSize hashlen, PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPBoolean				pgpSecPassphraseOK_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPByte const *passphrase,
							PGPSize pplen, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal );
PGPError				pgpKeyMaxSizes_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPUInt32 *maxEncryption,
							PGPUInt32 *maxDecryption, PGPUInt32 *maxSignature,
							PGPPublicKeyMessageFormat format );
PGPError				pgpSecProperties_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPBoolean *needsPassphrase,
							PGPBoolean *isSecretShared,
							PGPCipherAlgorithm *lockAlg, PGPUInt32 *lockBits );
PGPError				pgpFetchKeyInfo_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPByte **bufptr, PGPSize *bufsize );
PGPError				pgpOpenKeyDBFile_backRPC(PGPContextRef cdkContext,
							PGPOpenKeyDBFileOptions openFlags,
							PFLFileSpecRef pubFileRef,
							PFLFileSpecRef privFileRef, PGPUInt32 * kdbid,
							PGPUInt32 * numKeys, PGPUInt32 ** keyArray,
							PGPSize * keyArraySize );
PGPError				pgpNewKeyDB_backRPC( PGPContextRef context,
							PGPUInt32 *keydbid );
PGPError				pgpKeyDBArray_backRPC( PGPContextRef cdkContext,
							PGPUInt32 kdbid, PGPUInt32 * numKeys,
							PGPUInt32 ** keyArray, PGPSize * keyArraySize );
PGPError				pgpUpdateKeyDB_backRPC( PGPContextRef cdkContext,
							PGPUInt32 kdbid, PGPUInt32 * numNewKeys,
							PGPUInt32 ** newKeyArray,
							PGPSize * newKeyArraySize,
							PGPUInt32 ** changedkeylist,
							PGPSize *changedkeylistsize );
PGPError				pgpKeyDBFlush_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPUInt32 **changedkeylist,
							PGPSize *changedkeylistsize );
void					pgpFreeKeyDB_backRPC( PGPContextRef context,
							PGPUInt32 id );
PGPError				pgpSetKeyEnabled_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPBoolean enable );
PGPError				pgpSetKeyAxiomatic_backRPC( PGPContextRef context,
							PGPUInt32 id, PGPBoolean setAxiomatic,
							PGPBoolean checkPassphrase,
							char const *passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal );
PGPError				pgpPropagateTrust_backRPC( PGPContextRef context,
							PGPUInt32 setid, PGPUInt32 *keylist,
							PGPSize keylistsize, PGPUInt32 altid,
							PGPUInt32 const timenow,
							PGPUInt32 **changedkeylist,
							PGPSize *changedkeylistsize );
PGPError				pgpCheckKeyRingSigs_backRPC( PGPContextRef context,
							PGPUInt32 setid, PGPUInt32 *keylist,
							PGPSize keylistsize, PGPUInt32 altid,
							PGPBoolean checkAll, PGPUInt32 **changedkeylist,
							PGPSize *changedkeylistsize );
PGPError				pgpPrepareToCheckKeyRingSigs_backRPC(
							PGPContextRef context, PGPUInt32 setid,
							PGPUInt32 *keylist, PGPSize keylistsize,
							PGPUInt32 altid, PGPBoolean checkAll,
							PGPUInt32 *nsigs, PGPUInt32 **changedkeylist,
							PGPSize *changedkeylistsize );
PGPError				pgpCheckSig_backRPC( PGPContextRef context,
							PGPUInt32 setid, PGPUInt32 objid,
							PGPUInt32 altid, PGPBoolean checkAll,
							PGPBoolean revocationonly, PGPBoolean *handled,
							PGPBoolean *changed, PGPBoolean *verified );
PGPError				pgpDoGenerateKey_backRPC( PGPContextRef context,
							PGPUInt32 keydbID, PGPUInt32 masterkeyID,
							PGPByte pkalg, PGPUInt32 bits,
							PGPTime creationDate, PGPUInt16 expirationDays,
							char const * name, PGPSize name_len,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean passphraseIsKey,
							char const * masterpass, PGPSize masterpassLength,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPEventHandlerProcPtr progress,
							PGPUserValue userValue,
							PGPBoolean fastgen, PGPBoolean checkentropy,
							PGPBoolean useToken, PGPUInt32 tokenID,
							PGPUInt32 * adkkeylist, PGPSize adkkeylistsize,
							PGPByte adkclass, PGPUInt32 * rakkeylist,
							PGPSize rakkeylistsize, PGPByte rakclass,
							PGPCipherAlgorithm const * prefalg,
							PGPSize prefalgLength, PGPByte const * prefkeyserv,
							PGPSize prefkeyservLength, PGPUInt32 keyflags,
							PGPBoolean fkeyflags, PGPUInt32 keyservprefs,
							PGPBoolean fkeyservprefs, PGPUInt32 **newobjs,
							PGPSize *newobjsLength, PGPUInt32 * newKey);
PGPError				pgpCountTokens_backRPC (PGPContextRef context,
							PGPUInt32 *numTokens );
PGPError				pgpGetTokenInfo_backRPC (PGPContextRef context,
                            PGPUInt32   tokNumber, PGPTokenInfo *tokenInfo );
PGPError				pgpDeleteKeyOnToken_backRPC( PGPContextRef context, 
							PGPUInt32 keydbID, 
							const PGPKeyID *keyID, PGPUInt32 tokenNumber, 
							PGPByte const *pin, PGPSize pinLen );
PGPError				pgpWipeToken_backRPC (PGPContextRef context,
							PGPUInt32 tokNumber, PGPByte const *passphrase,
							PGPSize passphraseLength );
PGPError				pgpTokenPassphraseIsValid_backRPC (
							PGPContextRef context,
							PGPUInt32 tokNumber, PGPByte const *passphrase,
							PGPSize passphraseLength );
PGPError				pgpCopyKeyToToken_backRPC (PGPContextRef context,
							PGPUInt32 id, PGPUInt32 tokNumber,
							PGPBoolean isMaster,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, char const * PIN,
							PGPSize PINlength, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal);
PGPError                pgpTokenImportX509_backRPC( PGPContextRef context,
                            const PGPByte *keyID, 
                            const PGPByte *userID, PGPSize userIDlen, 
                            const PGPByte *x509, PGPSize x509len,
                            const PGPByte *password, PGPSize passwordLength );
PGPError                pgpTokenPutKeyContainer_backRPC( 
                            PGPContextRef context, const PGPByte *keyID, 
                            const PGPByte *password, PGPSize passwordSize, 
                            const PGPByte *cont, PGPSize contSize );
PGPError				pgpTokenGetKeyContainer_backRPC( PGPContextRef context,
							const PGPByte *keyID,
							const PGPByte *password, PGPSize passwordLen,		
							PGPByte	**contOut, PGPSize *contOutSize );
PGPError				pgpSetPKCS11DrvFile_backRPC( PGPByte *module );
PGPError				pgpAddUserID_backRPC (PGPContextRef context,
							PGPUInt32 id, PGPBoolean isAttribute,
							PGPAttributeType attributeType,
							char const * userIDData, PGPSize userIDLength,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPUInt32 **newobjs,
							PGPSize *newobjsLength );
PGPError				 pgpCertifyUserID_backRPC(PGPContextRef context,
							PGPUInt32 useridid, PGPUInt32 certifying_keyid,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPBoolean exportable,
							PGPTime creationDate, PGPUInt32 expiration,
							PGPByte trustDepth, PGPByte trustValue,
							char const * sRegExp, PGPSize sRegExpLength,
							PGPUInt32 **newobjs, PGPSize *newobjsLength );
PGPError				pgpCertifyPrimaryUserID_backRPC (PGPContextRef ctxt,
							PGPUInt32 useridid, char *passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 **newobjs, PGPSize *newobjsLength);
PGPError				pgpRevokeSig_backRPC (PGPContextRef context,
							PGPUInt32 sigid, char const * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 **newobjs, PGPSize *newobjsLength);
PGPError				pgpRevokeKey_backRPC (PGPContextRef context,
							PGPUInt32 key_id, char const * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 **newobjs, PGPSize *newobjsLength);
PGPError				pgpDoChangePassphrase_backRPC (PGPContextRef context,
							PGPUInt32 keydbid, PGPUInt32 key_id,
							PGPUInt32 masterkey_id, const char *oldphrase,
							PGPSize oldphraseLength, const char *newphrase,
							PGPSize newphraseLength,
							PGPBoolean newPassphraseIsKey,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal);
PGPError				pgpPassphraseIsValid_backRPC( PGPContextRef context,
							PGPUInt32 key_id, const char * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPBoolean * isValid);
PGPError				pgpPurgePassphraseCache_backRPC(PGPContextRef context);
PGPError				pgpSetKeyTrust_backRPC( PGPContextRef context,
							PGPUInt32 key_id, PGPUInt32 trust );
PGPError				pgpGetPasskeyBuffer_backRPC( PGPContextRef context,
							PGPUInt32 key_id, char const * passphrase,
							PGPSize passphraseLength, PGPByte **passkeyBuffer,
							PGPSize *passkeyBufferLength );
PGPError				pgpAddKeyOptions_backRPC (PGPContextRef context,
							PGPUInt32 key_id, char * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 *raklist, PGPSize raklistsize,
							PGPUInt32 rakclass, PGPUInt32 **newobjs,
							PGPSize *newobjsLength);
PGPError				pgpUpdateKeyOptions_backRPC (PGPContextRef context,
							PGPUInt32 key_id, char * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPCipherAlgorithm *prefalg,
							PGPSize prefalgLength, PGPByte *prefkeyserv,
							PGPSize prefkeyservLength, PGPUInt32 keyflags,
							PGPBoolean fkeyflags, PGPUInt32 keyservprefs,
							PGPBoolean fkeyservprefs, PGPUInt32 **newobjs,
							PGPSize *newobjsLength );
PGPError				pgpKeyDBAddObject_backRPC (PGPContextRef context,
							PGPUInt32 kdb_id, PGPUInt32 obj_id,
							PGPUInt32 **newobjs, PGPSize *newobjsLength,
							PGPUInt32 *pnewobj);
PGPError				pgpKeyDBRemoveObject_backRPC (PGPContextRef context,
							PGPUInt32 kdb_id, PGPUInt32 obj_id);
PGPError				pgpImportKeyBinary_backRPC(PGPContextRef cdkContext,
							PGPByte *buffer, PGPSize length,
							PGPUInt32 *kdbid, PGPUInt32 *numKeys,
							PGPUInt32 **keyArray, PGPSize *keyArraySize );
PGPError				pgpGlobalRandomPoolAddState_backRPC(
							PGPBoolean addKeyState, PGPInt32 keyEvent,
							PGPBoolean addMouseState,
							PGPBoolean addSystemState );
PGPError				pgpGlobalRandomPoolGetInfo_backRPC( PGPUInt32 *entropy,
							PGPUInt32 *size, PGPUInt32 *minEntropy,
							PGPBoolean *hasMinEntropy,
							PGPBoolean *hasIntelRNG );
PGPError				pgpRandomGetBytesEntropy_backRPC(
							PGPContextRef context, PGPUInt32 requestBytes,
							PGPUInt32 entropyBits, PGPByte **randBuf,
							PGPSize *randBufLen );
PGPError				pgpRandomAddBytes_backRPC( PGPContextRef context,
							PGPByte const *buf, PGPSize buflen );
PGPError				pgpRandomStir_backRPC( PGPContextRef context );
PGPError				pgpCopyKeys_backRPC( PGPContextRef context,
							PGPUInt32 srcid, PGPUInt32 dstid,
							PGPUInt32 *keylist, PGPSize keylistsize,
							PGPBoolean neednewkeylist, PGPUInt32 **newkeylist,
							PGPSize *newkeylistsize );
PGPError				pgpSetRandSeedFile_backRPC( PFLFileSpecRef randFile );
PGPError				pgpCreateKeypair_backRPC( PGPContextRef context,
							PGPUInt32 keydbID, PGPByte * seckeyBuf,
							PGPSize seckeySize, PGPByte * keyspecBuf,
							PGPSize keyspecSize, char const * name,
							PGPSize name_len,
							PGPUInt32 * adklist, PGPSize adklistsize,
							PGPByte adkclass,
							PGPUInt32 * raklist, PGPSize raklistsize,
							PGPByte rakclass,
							char const * passphrase,
							PGPSize passphraseLength, PGPBoolean
							passphraseIsKey, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPUInt32 **
							newobjs, PGPSize * newobjslength,
							PGPUInt32 * newkeyid );
PGPError				pgpCreateSubkeypair_backRPC( PGPContextRef context,
							PGPUInt32 masterkeyID, PGPByte *
							seckeyBuf, PGPSize seckeySize,
							PGPByte * keyspecBuf,
							PGPSize keyspecSize, char const * passphrase,
							PGPSize passphraseLength,
							PGPBoolean passphraseIsKey, PGPUInt32
							cacheTimeOut, PGPBoolean cacheGlobal,
							PGPUInt32 ** newobjs, PGPSize *
							newobjslength, PGPUInt32 * newkeyid );
							
#if PGP_MACINTOSH && PGPSDK_BACKEND
#pragma export off
#endif


/* pgpKeyDB.c */
						
PGPBoolean				pgpKeyDBIsMutable (PGPKeyDBRef kdb);
PGPBoolean				pgpKeyDBIsDirty (PGPKeyDBRef kdb);
PGPBoolean				pgpKeyDBUsesTokens (PGPKeyDBRef kdb);
void					pgpKeyDBChanged (PGPKeyDBRef kdb, PGPBoolean redolist);
PGPKeyDBRef				pgpKeyDBNextKeyDB (PGPKeyDBRef kdb);
PGPError				pgpKeyDBAddObject (PGPKeyDBRef kdb,
							PGPKeyDBObj *obj, PGPKeyDBObjRef *newobj);
PGPError				pgpKeyDBAddObject_internal (PGPKeyDBRef kdb,
							PGPKeyDBObj *obj, PGPKeyDBObjRef *pnewobj);
PGPError				pgpKeyDBRemoveObject (PGPKeyDBRef kdb,
							PGPKeyDBObj *obj);
PGPError				pgpKeyDBRemoveObject_internal (PGPKeyDBRef kdb,
							PGPKeyDBObj *obj);
PGPError				pgpKeyDBFlush_internal (PGPKeyDBRef kdb,
							PGPKeySetRef changedSet );
PGPBoolean				pgpKeyDBIsValid( PGPKeyDB const * keyDB);
void					pgpKeyDBDestroy_internal (PGPKeyDBRef kdb);
PGPKeySet *				pgpKeyDBPeekRootSet( PGPKeyDB *kdb );
PGPError				pgpKeyDBError( PGPKeyDB *kdb );
void					pgpKeyDBSetError( PGPKeyDB *kdb, PGPError err );
PGPBoolean				pgpFrontEndKeyDB( PGPKeyDBRef kdb );
void					pgpSetPendingNotify( PGPKeySetRef changeset,
							PGPKeyDBObjRef changeobj, PGPKeySetRef newset,
							PGPKeyDBObjRef newobj );
PGPError				pgpUpdateKeyDB_internal( PGPKeyDB *db,
							PGPKeySetRef newSet, PGPKeySetRef changeSet );
void					pgpSetBackendUpdateNeeded( void );
void					pgpKeyDBBackendUpdate( void );
PGPKeyDBRef				pgpCreateKeyDBFromKeyRings (PGPContextRef context,
							PFLFileSpecRef pubFileRef,
							PFLFileSpecRef privFileRef,
							PGPFlags openFlags, PGPError *error);
PGPKeyDBRef				pgpCreateKeyDBFromMemory (PGPContextRef context,
							void *buf, PGPSize buflen, PGPError *error);
PGPError				pgpNewKeyDB_internal( PGPContextRef context,
							PGPKeyDBRef *keydb );
PGPError				pgpNewFrontEndKeyDB( PGPContextRef context,
							PGPUInt32 kdbid, PGPUInt32 *keyArray,
							PGPUInt32 keyCount, PGPKeyDBRef *keydb );


/* pgpKeyAdd.c */

PGPByte *				pgpKeyDBObjToPubData( PGPKeyDBObj *key, PGPSize *plen);
PGPKeyDBObj *			pgpFindMatchingChild( PGPKeyDBObj *parent,
							PGPByte const *buf, PGPSize len );
PGPKeyDBObj *
						pgpFindPairedX509Key( PGPKeyDBObj *cakey,
							PGPKeyDBRef kdb );
PGPKeyDBObj *			pgpAddKey( PGPInt32 pktbyte, PGPByte const *buf,
							PGPSize len, PGPKeyDB *kdb, PGPKeyDBObj *parent,
							PGPBoolean trusted, PGPError *err);
PGPKeyDBObj *			pgpAddUserID( PGPInt32 pktbyte, PGPByte const *buf,
							PGPSize len, PGPKeyDB *kdb, PGPKeyDBObj *parent,
							PGPError *err );
PGPKeyDBObj *			pgpAddSig( PGPInt32 pktbyte, PGPByte const *buf,
							PGPSize len, PGPKeyDB *kdb, PGPKeyDBObj *parent,
							PGPError *err);
PGPKeyDBObj *			pgpAddCRL( PGPInt32 pktbyte, PGPByte const *buf,
							PGPSize len, PGPKeyDB *kdb, PGPKeyDBObj *parent,
							PGPError *err);
PGPKeyDBObj *			pgpAddUnknown( PGPInt32 pktbyte, PGPByte const *buf,
							PGPSize len, PGPKeyDB *kdb, PGPKeyDBObj *parent,
							PGPError *err );
PGPKeyDBObj *			pgpAddObject( PGPByte const *buf, PGPSize len,
							PGPKeyDB *kdb, PGPKeyDBObj *parent,
							PGPBoolean trusted, PGPError *err );
PGPError				pgpKeyCleanup( PGPKeyDBObj *key, PGPBoolean trusted );
void					pgpCopyObjectTrust( PGPKeyDBObj *oldobj,
							PGPKeyDBObj *newobj );
PGPError				pgpReadKeyFile (PGPKeyDB *kdb, PGPFile *f,
							PGPBoolean trusted);
PGPKeyDBObj *			pgpCreateKeypair (PGPEnv const *env, PGPKeyDB *db,
							PGPSecKey *seckey, PGPKeySpec *keyspec,
							char const *name, size_t namelen,
							PGPKeySet const *adkset, PGPByte adkclass,
							PGPKeySet const *rakset, PGPByte rakclass,
							PGPByte const *passphrase, PGPSize pplen,
							PGPBoolean hashedPhrase,
							PGPError *error);
PGPKeyDBObj *			pgpCreateSubkeypair (PGPKeyDBObj *keyobj,
							PGPEnv const *env,
							PGPSecKey *subseckey, PGPKeySpec *keyspec,
							PGPByte const *passphrase, PGPSize pplen,
							PGPBoolean hashedPhrase, PGPError *error);
PGPError				pgpSignObject( PGPKeyDBObj *obj, PGPSigSpec *spec );
PGPKeyDBObj *			pgpCreateKey(PGPKeyDB *db, PGPKeyDBObj *parent,
							PGPBoolean secret, PGPPubKey const *pkey,
							PGPSecKey const *skey, PGPKeySpec const *ks,
							PGPByte pkalg);
PGPKeyDBObj *			pgpCreateUserID( PGPKeyDB *db, PGPKeyDBObj *key,
							PGPByte const *data, PGPSize len );
PGPKeyDBObj *			pgpCreateAttribute( PGPKeyDB *db, PGPKeyDBObj *key,
							PGPByte attributeType, PGPByte const *data,
							PGPSize len );
PGPKeyDBObj *			pgpCreateSig( PGPKeyDB *db, PGPKeyDBObj *parent,
							PGPByte const *sig, PGPSize len );
PGPKeyDBObj *			pgpCreateCRL( PGPKeyDB *db, PGPKeyDBObj *parent,
							PGPByte const *crl, PGPSize len );
PGPError				pgpUpdateSecKeyData( PGPKeyDBObj *key,
							PGPSecKey const *skey );
PGPError				pgpAddFromKeyArray( PGPKeyDB *kdb, PGPKeyDBObj *parent,
							PGPUInt32 *keyArray, PGPUInt32 keyCount,
							PGPBoolean newobjs );


/* pgpKeyFilter.c */

PGPBoolean				pgpKeyDBObjMatchesFilter(PGPFilterRef filter,
							PGPKeyDBObjRef obj);


/* pgpKeyID.c */

PGPBoolean				pgpKeyIDIsValid( PGPKeyID const * id );
PGPError				pgpNewKeyIDFromRawData(const void * data,
							PGPByte pkalg, PGPSize dataSize,
							PGPKeyID * outID );
PGPByte const *			pgpGetKeyIDBytes( PGPKeyID const *keyID );
PGPBoolean				pgpKeyIDsEqual(PGPKeyID const * key1,
							PGPKeyID const * key2);
PGPUInt32				pgpKeyIDLength( PGPKeyID const *key1 );


/* pgpKeyIO.c */

PGPError				pgpImportKeyBinary (PGPContextRef cdkContext,
							PGPByte * buffer, size_t length,
							PGPKeyDBRef * outRef);
PGPError				pgpImportKeyBinary_internal (PGPContextRef cdkContext,
							PGPByte * buffer, size_t length,
							PGPKeyDBRef * outRef);
PGPError				pgpImportKeySetInternal (PGPContextRef context,
							PGPKeyDBRef *keydb, PGPOptionListRef optionList);
PGPError				pgpExportToPGPFile( PGPKeySet *set, PGPFile *file,
							PGPBoolean exportmastersecrets,
							PGPBoolean exportsubsecrets,
							PGPBoolean includeattributes, 
							PGPBoolean includepubkey );
PGPError				pgpExportKeySetInternal (PGPKeySet *keys,
							PGPOptionListRef optionList);
PGPError				pgpExportInternal (PGPContextRef context,
							PGPOptionListRef optionList);
PGPError				pgpOpenKeyDBFile_internal( PGPContextRef cdkContext,
							PGPOpenKeyDBFileOptions openFlags,
							PFLFileSpecRef pubFileRef,
							PFLFileSpecRef privFileRef,
							PGPKeyDBRef * keyDBOut );
PGPError				pgpCreateX509CertificateInternal(PGPKeyDBObjRef signer,
							PGPKeyDBObjRef targetkey, PGPKeyDBObjRef *sig,
							PGPOptionListRef optionList );


/* pgpKeyIter.c */

PGPError				pgpGetFirstSubKey(PGPKeyDBObjRef key,
							PGPKeyDBObjRef * outRef );
PGPBoolean				pgpKeyIterIsValid( PGPKeyIter const * keyIter);
PGPError				pgpKeyIterNextObject(PGPKeyIterRef iter,
							PGPKeyDBObjRef *outRef);

/* pgpKeyList.c */

PGPBoolean				pgpKeyListIsValid( PGPKeyList const * keyList);
PGPError				pgpUpdateList(PGPKeyList *list);


/* pgpKeyMan.c */

PGPError				pgpRevokeKeyInternal(PGPKeyDBObjRef key,
							PGPOptionListRef optionList );
PGPError				pgpRevokeKey_internal (PGPKeyDBObjRef key,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal);
PGPUInt32				pgpKeyEntropyNeededInternal(PGPContextRef context,
							PGPOptionListRef optionList );
PGPBoolean				pgpPassphraseIsValidInternal(PGPKeyDBObjRef key,
							PGPOptionListRef optionList);
PGPError				pgpPassphraseIsValid_internal(PGPKeyDBObjRef key,
							const char * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPBoolean * isValid);
PGPError				pgpDoGenerateKey_internal( PGPKeyDB * keyDB,
							PGPKeyDBObj * masterkey, PGPByte pkalg,
							PGPUInt32 bits, PGPTime creationDate,
							PGPUInt16 expirationDays, char const * name,
							int name_len, char const * passphrase,
							PGPSize passphraseLength,
							PGPBoolean passphraseIsKey,
							char const * masterpass, PGPSize masterpassLength,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPEventHandlerProcPtr progress,
							PGPUserValue userValue, PGPBoolean fastgen,
							PGPBoolean checkentropy, PGPBoolean useToken,
							PGPUInt32 tokenID, PGPKeySet const *adkset,
							PGPByte adkclass, PGPKeySet const *rakset,
							PGPByte rakclass,
							PGPCipherAlgorithm const * prefalg,
							PGPSize prefalgLength,
							PGPByte const * prefkeyserv,
							PGPSize prefkeyservLength, PGPUInt32 keyflags,
							PGPBoolean fkeyflags, PGPUInt32 keyservprefs,
							PGPBoolean fkeyservprefs, PGPKeyDBObj ** newkey);
PGPError				pgpGenerateKeyInternal(PGPKeyDBObjRef *key,
							PGPOptionListRef optionList);
PGPError				pgpGenerateSubKeyInternal(PGPKeyDBObjRef *subkey,
							PGPOptionListRef optionList);
PGPError				pgpAddKeyOptionsInternal (PGPKeyDBObjRef key,
							PGPOptionListRef optionList);
PGPError				pgpAddKeyOptions_internal (PGPKeyDBObjRef key,
							char * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPKeySetRef rakset,
							PGPUInt32 rakclass);
PGPError				pgpRemoveKeyOptionsInternal (PGPKeyDBObjRef key,
							PGPOptionListRef optionList);
PGPError				pgpUpdateKeyOptionsInternal (PGPKeyDBObjRef key,
							PGPOptionListRef optionList);
PGPError				pgpUpdateKeyOptions_internal (PGPKeyDBObjRef key,
							char * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPCipherAlgorithm *prefalg,
							PGPSize prefalgLength, PGPByte *prefkeyserv,
							PGPSize prefkeyservLength, PGPUInt32 keyflags,
							PGPBoolean fkeyflags, PGPUInt32 keyservprefs,
							PGPBoolean fkeyservprefs);
PGPError				pgpDisableKey (PGPKeyDBObj *key);
PGPError				pgpEnableKey (PGPKeyDBObj *key);
PGPError				pgpDoChangePassphrase_internal (PGPKeyDB *keyDB,
							PGPKeyDBObj *key, PGPKeyDBObj *masterkey, 
							const char *oldphrase, PGPSize oldphraseLength,
							const char *newphrase, PGPSize newphraseLength,
							PGPBoolean newPassphraseIsKey,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal);
PGPError				pgpChangePassphraseInternal(PGPKeyDBObjRef key,
							PGPOptionListRef optionList);
PGPError				pgpRevokeSubKey_internal (PGPKeyDBObjRef subkey,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal);
PGPError				pgpChangeSubKeyPassphraseInternal(
							PGPKeyDBObjRef subkey,PGPOptionListRef optionList);
PGPError				pgpRevokeSubKeyInternal(PGPKeyDBObjRef subkey,
							PGPOptionListRef optionList );
PGPError				pgpGetKeyPasskeyBufferInternal (PGPKeyDBObjRef key,
							void *passkeyBuffer, PGPOptionListRef optionList);
PGPError				pgpGetSubKeyPasskeyBufferInternal (
							PGPKeyDBObjRef subkey, void *passkeyBuffer,
							PGPOptionListRef optionList);
PGPError				pgpGetPasskeyBuffer_internal (PGPKeyDBObjRef key,
							char const * passphrase, PGPSize passphraseLength,
							PGPByte *passkeyBuffer);
PGPError				pgpAddUserIDInternal(PGPKeyDBObjRef key,
							char const * userID, PGPOptionListRef optionList);
PGPError				pgpAddUserID_internal (PGPKeyDBObjRef key,
							PGPBoolean isAttribute,
							PGPAttributeType attributeType,
							char const * userIDData, PGPSize userIDLength,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal );
PGPError				pgpAddAttributeInternal(PGPKeyDBObjRef key,
							PGPAttributeType attributeType,
							PGPByte const *attributeData,
							PGPSize attributeLength,
							PGPOptionListRef optionList);
PGPError				pgpCertifyPrimaryUserIDInternal (PGPKeyDBObj * userid,
							PGPOptionListRef optionList);
PGPError				pgpCertifyPrimaryUserID_internal (PGPKeyDBObj *userid,
							char *passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal );
PGPError				pgpCertifyUserIDInternal (PGPKeyDBObj * userid,
							PGPKeyDBObj * certifying_key,
							PGPOptionListRef optionList);
PGPError				 pgpCertifyUserID_internal(PGPKeyDBObj * userid,
							PGPKeyDBObjRef certifying_key,
							char const * passphrase, PGPSize passphraseLength,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPBoolean exportable,
							PGPTime creationDate, PGPUInt32 expiration,
							PGPByte trustDepth, PGPByte trustValue,
							char const * sRegExp );
PGPError				pgpGetSigCertifierKey(PGPKeyDBObjRef sig,
							PGPKeyDBRef otherdb, PGPKeyDBObj ** certkey);
PGPError				pgpRevokeSigInternal(PGPKeyDBObjRef sig,
							PGPOptionListRef optionList);
PGPError				pgpRevokeSig_internal (PGPContextRef context,
							PGPKeyDBObjRef sig, char const * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal);
PGPError				pgpSetKeyAxiomaticInternal(PGPKeyDBObjRef key,
							PGPBoolean isAxiomatic,
							PGPOptionListRef optionList);
PGPError				pgpSetKeyAxiomatic_internal (PGPKeyDBObj * key,
							PGPBoolean setAxiomatic,
							PGPBoolean checkPassphrase, char const *passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal);
PGPError				pgpSetKeyTrust_internal( PGPKeyDBObj *key,
							PGPUInt32 trust );
PGPError				pgpUnsetKeyAxiomatic (PGPKeyDBObj *key);
PGPError				pgpGetKeyNumberInternal (PGPKeyDBObj *key,
							PGPKeyDBObjProperty propname, PGPInt32 *prop);
PGPError				pgpGetSubKeyNumberInternal (PGPKeyDBObjRef subkey,
							PGPKeyDBObjProperty propname, PGPInt32 *prop);
PGPError				pgpGetUserIDNumberInternal (PGPKeyDBObj *userid,
							PGPKeyDBObjProperty propname, PGPInt32 *prop);
PGPError				pgpGetUserIDBooleanInternal (PGPKeyDBObj *userid,
							PGPKeyDBObjProperty propname, PGPBoolean *prop);
PGPError				pgpGetSigNumberInternal (PGPKeyDBObj *sig,
							PGPKeyDBObjProperty propname, PGPInt32 *prop);
PGPError				pgpGetKeyIDOfCertifier(PGPKeyDBObj *sig,
							PGPKeyID *outID );
PGPError				pgpGetSigTimeInternal (PGPKeyDBObj *sig,
							PGPKeyDBObjProperty propname, PGPTime *prop);
PGPError				pgpGetSigBooleanInternal (PGPKeyDBObj *sig,
							PGPKeyDBObjProperty propname, PGPBoolean *prop);
PGPContextRef			pgpGetKeyDBContext( PGPKeyDBRef ref );
PGPError				pgpGetKeyBoolean( PGPKeyDBObjRef key,
							PGPKeyDBObjProperty propname, PGPBoolean *prop);
PGPError				pgpGetKeyNumber( PGPKeyDBObjRef key,
							PGPKeyDBObjProperty propname, PGPInt32 *prop);
PGPError				pgpGetKeyPropertyBuffer( PGPKeyDBObjRef key,
							PGPKeyDBObjProperty propname, PGPSize bufferSize,
							void *data, PGPSize *fullSize);
PGPError				pgpGetKeyAllocatedPropertyBuffer( PGPKeyDBObjRef key,
							PGPKeyDBObjProperty propname, PGPByte **allocBuf,
							PGPSize *fullSize);
PGPError				pgpGetKeyTime( PGPKeyDBObjRef key,
							PGPKeyDBObjProperty propname, PGPTime *prop);
PGPError				pgpGetSubKeyBoolean( PGPKeyDBObjRef subkey,
							PGPKeyDBObjProperty propname, PGPBoolean *prop);
PGPError				pgpGetSubKeyNumber( PGPKeyDBObjRef subkey,
							PGPKeyDBObjProperty propname, PGPInt32 *prop);
PGPError				pgpGetSubKeyPropertyBuffer( PGPKeyDBObjRef subkey,
							PGPKeyDBObjProperty propname, PGPSize bufferSize,
							void *data, PGPSize *fullSize);
PGPError				pgpGetSubKeyAllocatedPropertyBuffer(
							PGPKeyDBObjRef subkey,
							PGPKeyDBObjProperty propname, PGPByte **allocBuf,
							PGPSize *fullSize);
PGPError				pgpGetSubKeyTime( PGPKeyDBObjRef subkey,
							PGPKeyDBObjProperty propname, PGPTime *prop);
PGPError				pgpGetUserIDNumber( PGPKeyDBObjRef userID,
							PGPKeyDBObjProperty propname, PGPInt32 *prop);
PGPError				pgpGetUserIDBoolean( PGPKeyDBObjRef userID,
							PGPKeyDBObjProperty propname, PGPBoolean *prop);
PGPError				pgpGetUserIDStringBuffer( PGPKeyDBObjRef userID,
							PGPKeyDBObjProperty propname, PGPSize bufferSize,
							char * string, PGPSize * fullSize);
PGPError				pgpGetUserIDAllocatedStringBuffer(
							PGPKeyDBObjRef userID,
							PGPKeyDBObjProperty propname, PGPByte **allocBuf,
							PGPSize * fullSize);
PGPError				pgpGetSigPropertyBuffer( PGPKeyDBObjRef cert,
							PGPKeyDBObjProperty propname, PGPSize bufferSize,
							void *data, PGPSize *fullSize);
PGPError				pgpGetSigAllocatedPropertyBuffer( PGPKeyDBObjRef cert,
							PGPKeyDBObjProperty propname, PGPByte **allocBuf,
							PGPSize *fullSize);
PGPError				pgpGetSigBoolean( PGPKeyDBObjRef cert,
							PGPKeyDBObjProperty propname, PGPBoolean *prop);
PGPError				pgpGetSigNumber( PGPKeyDBObjRef cert,
							PGPKeyDBObjProperty propname, PGPInt32 *prop);
PGPError				pgpGetSigTime( PGPKeyDBObjRef cert,
							PGPKeyDBObjProperty propname, PGPTime *prop);
PGPError				pgpGetKeyDBObjNumber( PGPKeyDBObj *obj,
							PGPKeyDBObjProperty propname, PGPInt32 *prop);
PGPError				pgpCountTokens_internal( PGPContextRef context,
							PGPUInt32 *numTokens );
PGPError				pgpGetTokenInfo_internal( PGPContextRef context,
							PGPUInt32 tokNumber, PGPTokenInfo *tokenInfo );
PGPError				pgpDeleteKeyOnToken_internal( PGPContextRef context,
							PGPUInt32 keydbID, const PGPKeyID *keyID,
							PGPUInt32 toknumber,
							PGPByte const *pin, PGPSize pinLen );
PGPError				pgpWipeToken_internal( PGPContextRef context,
							PGPUInt32 tokenID, PGPByte const *passphrase,
							PGPSize passphraseLength );
PGPError				pgpTokenPassphraseIsValid_internal(
							PGPContextRef context,
							PGPUInt32 tokenID, PGPByte const *passphrase,
							PGPSize passphraseLength );
PGPError				pgpCopyKeyToToken_internal(PGPKeyDBObjRef key,
							PGPUInt32 tokNumber, PGPBoolean isMaster, char const * passphrase,
							PGPSize passphraseLength, PGPBoolean hashedPhrase,
							char const *PIN, PGPSize PINlength,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal);
PGPError				pgpCopyKeyToToken( PGPKeySet *keyset, PGPOptionListRef optionList );
PGPError                pgpTokenImportX509_internal(
	                        PGPContextRef context, const PGPByte *keyID,
                            const PGPByte *userID, PGPSize userIDlen,
                            const PGPByte *x509, PGPSize x509len,
                            const PGPByte *password, PGPSize passwordLength );
PGPError                pgpTokenPutKeyContainer_internal(
                            PGPContextRef context, const PGPByte *keyID, 
                            const PGPByte *password, PGPSize passwordSize, 
                            const PGPByte *cont, PGPSize contSize );
PGPError				pgpTokenGetKeyContainer_internal( PGPContextRef context,
							const PGPByte *keyID,
							const PGPByte *password, PGPSize passwordSize, 
							PGPByte **contOut, PGPSize *contOutSize );

/* pgpKeyObj.c */

PGPUInt32				pgpObjectType(PGPKeyDBObj const *obj);
void					pgpObjectHold(PGPKeyDBObj *obj);
void					pgpObjectRelease(PGPKeyDBObj *obj);
PGPBoolean				pgpFrontEndKey( PGPKeyDBObj const *obj );
PGPUInt32				pgpKeyDBObjID( PGPKeyDBObj const *obj );
PGPByte const *			pgpFetchObject( PGPKeyDBObj const *obj, PGPSize *len );
void *					pgpNewObjData( PGPKeyDB *kdb, PGPByte const *buf,
							PGPSize len );
PGPError				pgpUpdateObjData( PGPKeyDBObj *obj, PGPByte *data,
							PGPSize len );
PGPUInt32				pgpKeyBits(PGPKeyDBObj *key);
PGPUInt32				pgpKeyCreation(PGPKeyDBObj *key);
PGPUInt32				pgpKeyExpiration(PGPKeyDBObj *key);
PGPUInt32				pgpKeyUseInternal(PGPKeyDBObj *key,
							PGPBoolean unExpired, PGPBoolean revokedOK);
PGPByte					pgpKeyTrust(PGPKeyDBObj *key);
void					pgpKeySetTrust(PGPKeyDBObj *key, PGPByte trust);
void					pgpKeySetAxiomatic(PGPKeyDBObj *key);
void					pgpKeyResetAxiomatic (PGPKeyDBObj *key);
PGPBoolean				pgpKeyAxiomatic(PGPKeyDBObj *key);
PGPBoolean				pgpKeyIsSubkey (PGPKeyDBObj const *key);
PGPBoolean				pgpKeyDisabled(PGPKeyDBObj *key);
void					pgpKeyDisable(PGPKeyDBObj *key);
void					pgpKeyEnable(PGPKeyDBObj *key);
PGPBoolean				pgpKeyRevoked(PGPKeyDBObj *key);
void					pgpKeyOnToken(PGPKeyDBObj *key);
void					pgpKeyOffToken(PGPKeyDBObj *key);
PGPBoolean				pgpKeyIsOnToken(PGPKeyDBObj *key);
void					pgpSecSetValidated(PGPKeyDBObj *key);
void					pgpSecClearValidated(PGPKeyDBObj *key);
PGPBoolean				pgpSecIsValidated(PGPKeyDBObj *key);
PGPBoolean				pgpSigRevoked (PGPKeyDBObj *sig);
PGPUInt16				pgpKeyConfidence(PGPKeyDBObj *key);
void					pgpKeyID8(PGPKeyDBObj const *key, PGPByte *pkalg,
							PGPKeyID *keyID);
void					pgpKeyID4(PGPKeyDBObj const *key, PGPByte *pkalg,
							PGPKeyID *keyID);
PGPBoolean				pgpKeyV3(PGPKeyDBObj const *key);
PGPError				pgpKeyFingerprint16(PGPKeyDBObj *key, PGPByte *buf);
PGPError				pgpKeyFingerprint20(PGPKeyDBObj *key, PGPByte *buf);
PGPError				pgpKeyFingerprint20n(PGPKeyDBObj *key, PGPByte *buf);
PGPBoolean				pgpKeyIsSec(PGPKeyDBObj *key);
PGPKeyDBObj *			pgpKeySubkey(PGPKeyDBObj const *key,
							PGPKeySet *set);
PGPKeyDBObj *			pgpKeyMasterkey (PGPKeyDBObj const *subkey);
PGPByte const *			pgpKeyMPIData( PGPKeyDBObj const *key, PGPSize *plen );
PGPPubKey *				pgpKeyPubKey(PGPKeyDBObj *key, int use);
PgpVersion				pgpKeyVersion (PGPKeyDBObj *key);
PGPSecKey *				pgpSecSecKey(PGPKeyDBObj *sec, int use);
PGPKeyDBObj *			pgpLatestSigByKey (PGPKeyDBObj const *obj,
							PGPKeyDBObj const *key);
PGPBoolean				pgpSigIsSuperceded( PGPKeyDBObj const *sig );
char const *			pgpUserIDName(PGPKeyDBObj *name, PGPSize *lenp);
PGPBoolean				pgpUserIDIsAttribute(PGPKeyDBObj *name);
PGPUInt32				pgpUserIDCountAttributes(PGPKeyDBObj *name);
PGPByte const *			pgpUserIDAttributeSubpacket (PGPKeyDBObj *name,
							PGPUInt32 nth, PGPUInt32 *subpacktype,
							PGPSize *plen, PGPError *error);
PGPBoolean				pgpUserIDIsSelfSigned( PGPKeyDBObj *name,
							PGPBoolean checked);
PGPKeyDBObj *			pgpKeyPrimaryUserID (PGPKeyDBObj *key, PGPUInt32 type);
PGPError				pgpKeySetPrimaryUserID( PGPKeyDBObj *userid );
PGPError				pgpKeyClearPrimaryUserIDs( PGPKeyDBObj *key,
							PGPUInt32 type );
PGPByte					pgpUserIDOldValidity(PGPKeyDBObj *name);
PGPBoolean				pgpUserIDWarnonly(PGPKeyDBObj *name);
void					pgpUserIDSetWarnonly(PGPKeyDBObj *name);
PGPUInt16				pgpUserIDValidity(PGPKeyDBObj *name);
PGPUInt16				pgpUserIDConfidence(PGPKeyDBObj *name);
PGPBoolean				pgpUserIDConfidenceUndefined(PGPKeyDBObj *name);
void					pgpUserIDSetConfidence(PGPKeyDBObj *name,
							PGPUInt16 confidence);
PGPKeyDBObj *			pgpSigMaker(PGPKeyDBObj const *sig);
PGPKeyDBObj *			pgpSigMakerDummyOK(PGPKeyDBObj const *sig);
void					pgpSigID8(PGPKeyDBObj const *sig, PGPByte *pkalg,
							PGPKeyID *keyID);
PGPByte					pgpSigTrust(PGPKeyDBObj *sig);
PGPBoolean				pgpSigChecked(PGPKeyDBObj const *sig);
PGPBoolean				pgpSigTried(PGPKeyDBObj const *sig);
PGPBoolean				pgpSigExportable(PGPKeyDBObj const *sig);
PGPByte					pgpSigTrustLevel(PGPKeyDBObj const *sig);
PGPByte					pgpSigTrustValue(PGPKeyDBObj const *sig);
PGPUInt16				pgpSigConfidence(PGPKeyDBObj *sig);
PGPUInt32				pgpSigType(PGPKeyDBObj const *sig);
PGPUInt32				pgpSigTimestamp(PGPKeyDBObj const *sig);
PGPUInt32				pgpSigExpiration(PGPKeyDBObj const *sig);
PGPBoolean				pgpSigIsSelfSig(PGPKeyDBObj const *sig);
PGPBoolean				pgpSigIsX509(PGPKeyDBObj const *sig);
PGPByte *				pgpSigX509Certificate(PGPKeyDBObj *sig, PGPSize *len);
PGPBoolean				pgpCRLChecked(PGPKeyDBObj const *crl);
PGPBoolean				pgpCRLTried(PGPKeyDBObj const *crl);
PGPUInt32				pgpCRLCreation(PGPKeyDBObj const *crl);
PGPUInt32				pgpCRLExpiration(PGPKeyDBObj const *crl);
PGPBoolean				pgpKeyHasCRL(PGPKeyDBObj *key);
PGPKeyDBObj *			pgpKeyNthCRL(PGPKeyDBObj *key, PGPUInt32 n,
							PGPUInt32 *crlcount);
PGPKeyDBObj *			pgpKeyEarliestCRL(PGPKeyDBObj *key,
							PGPBoolean expiration);
PGPBoolean				pgpCRLIsCurrent (PGPKeyDBObj *crl, PGPUInt32 tstamp);
PGPByte const *			pgpCRLDistributionPoint( PGPKeyDBObj *crl,
							PGPSize *len );
PGPError				pgpListCRLDistributionPoints(PGPMemoryMgrRef mgr,
							PGPKeyDBObj *key, PGPUInt32 *pnDistPoints,
							PGPByte **pbuf, PGPSize **pbufsizes);
PGPByte const *			pgpKeyFindSubpacket (PGPKeyDBObj *obj, int subpacktype,
							unsigned nth, PGPSize *plen, PGPBoolean *pcritical,
							PGPBoolean *phashed, PGPUInt32 *pcreation,
							PGPUInt32 *pmatches, PGPError *error);
PGPKeyDBObj *			pgpKeyAdditionalRecipientRequestKey (PGPKeyDBObj *obj,
							unsigned nth, PGPByte *pkalg, PGPKeyID *keyid,
							PGPByte *pclass, unsigned *pkeys, PGPError *error);
PGPKeyDBObj *			pgpKeyRevocationKey (PGPKeyDBObj *obj, unsigned nth,
							PGPByte *pkalg, PGPKeyID *keyid, PGPByte *pclass,
							unsigned *pkeys, PGPError *error);
PGPBoolean				pgpKeyIsRevocationKey (PGPKeyDBObj *key,
							PGPKeyDBObj *rkey);
PGPBoolean				pgpKeyHasThirdPartyRevocation (PGPKeyDBObj *obj,
							PGPKeyDBObj **revkey, PGPByte *pkalg,
							PGPKeyID *keyid, PGPError *error);
PGPSigSpec *			pgpSigSigSpec (PGPKeyDBObj *sig, PGPError *error);
PGPBoolean				pgpKeyDBHasExpiringObjects(PGPKeyDB *db,
							PGPTime time1, PGPTime time2 );
PGPKeyDBObj *			pgpNewObject(PGPKeyDB *db, PGPUInt32 objtype);
void					pgpFreeObject(PGPKeyDB *db, PGPKeyDBObj *obj);
PGPKeyDBObj *			pgpKeyDBFindKey(PGPKeyDB const *db, PGPByte pkalg,
							PGPByte const keyID[8]);
PGPKeyDBObj *			pgpKeyDBFindKey20n(PGPKeyDB *db, PGPByte const *fp20n);
void					pgpKeyDBListSigsBy(PGPKeyDB *db);
PGPBoolean				pgpSubkeyValid(PGPKeyDBObj *subkey,
							PGPBoolean unExpired, PGPBoolean revokedO);
PGPError				pgpFingerprint20HashBuf(PGPContextRef context,
							PGPByte const *buf, size_t len, PGPByte *hash);
void *					pgpSigRegexp( PGPKeyDBObj const *sig );
PGPBoolean				sigRevokesKey (PGPKeyDBObj *sig);
PGPKeyDBObj *			pgpKeyById8(PGPKeyDBRef db, PGPByte pkalg,
							PGPByte const * keyIDbytes);
PGPError				pgpGetKeyByKeyID(PGPKeyDBRef db,
							PGPKeyID const *keyIDIn,
							PGPBoolean dummyOK, PGPBoolean deletedOK,
							PGPKeyDBObjRef * outRef );
PGPError				pgpGetKeyByKeyID_internal(PGPKeyDBRef db,
							PGPKeyID const *keyIDIn,
							PGPBoolean dummyOK, PGPBoolean deletedOK,
							PGPKeyDBObjRef * outRef );
void					pgpLinkKeyToTree( PGPKeyDBObj *key, PGPKeyDB *kdb );
PGPError				pgpKeyEncrypt( PGPKeyDBObj *key,
							PGPByte const *inbuf, PGPSize inbuflen,
							PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPError				pgpKeyDecrypt( PGPKeyDBObj *key,
							PGPByte const *passphrase, PGPSize pplen,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPByte const *inbuf,
							PGPSize inbuflen,
							PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPInt32				pgpKeyVerify( PGPKeyDBObj *key, PGPByte const *inbuf,
							PGPSize inbuflen, PGPHashAlgorithm hashalg,
							PGPByte const *hash, PGPSize hashlen, 
							PGPPublicKeyMessageFormat format );
PGPError				pgpKeySign( PGPKeyDBObj *key,
							PGPByte const *passphrase, PGPSize pplen,
							PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPHashAlgorithm hashalg, PGPByte const *hash,
							PGPSize hashlen, PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPBoolean				pgpSecPassphraseOK( PGPKeyDBObj *key,
							PGPByte const *passphrase, PGPSize pplen,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal);
PGPError				pgpKeyMaxSizes( PGPKeyDBObj *key,
							PGPUInt32 *maxEncryption, PGPUInt32 *maxDecryption,
							PGPUInt32 *maxSignature,
							PGPPublicKeyMessageFormat format );
PGPError				pgpSecProperties( PGPKeyDBObj *key,
							PGPBoolean *needsPassphrase,
							PGPBoolean *isSecretShared,
							PGPCipherAlgorithm *lockAlg, PGPUInt32 *lockBits );
PGPError				pgpKeyEncrypt_internal( PGPKeyDBObj *key,
							PGPByte const *inbuf, PGPSize inbuflen,
							PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPError				pgpKeyDecrypt_internal( PGPKeyDBObj *key,
							PGPByte const *passphrase, PGPSize pplen,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal, PGPByte const *inbuf,
							PGPSize inbuflen,
							PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPInt32				pgpKeyVerify_internal( PGPKeyDBObj *key,
							PGPByte const *inbuf, PGPSize inbuflen,
							PGPHashAlgorithm hashalg, PGPByte const *hash,
							PGPSize hashlen, PGPPublicKeyMessageFormat format);
PGPError				pgpKeySign_internal( PGPKeyDBObj *key,
							PGPByte const *passphrase, PGPSize pplen,
							PGPBoolean hashedPhrase,
							PGPUInt32 cacheTimeOut, PGPBoolean cacheGlobal,
							PGPHashAlgorithm hashalg, PGPByte const *hash,
							PGPSize hashlen, PGPPublicKeyMessageFormat format,
							PGPByte **outbuf, PGPSize *outbuflen );
PGPBoolean				pgpSecPassphraseOK_internal( PGPKeyDBObj *key,
							PGPByte const *passphrase, PGPSize pplen,
							PGPBoolean hashedPhrase, PGPUInt32 cacheTimeOut,
							PGPBoolean cacheGlobal);
PGPError				pgpKeyMaxSizes_internal( PGPKeyDBObj *key,
							PGPUInt32 *maxEncryption, PGPUInt32 *maxDecryption,
							PGPUInt32 *maxSignature,
							PGPPublicKeyMessageFormat format );
PGPError				pgpSecProperties_internal( PGPKeyDBObj *key,
							PGPBoolean *needsPassphrase,
							PGPBoolean *isSecretShared,
							PGPCipherAlgorithm *lockAlg, PGPUInt32 *lockBits );
PGPError				pgpMarkKeyDBObjectDeleted( PGPKeyDBObj *obj );
PGPBoolean				pgpKeyDBObjIsDeleted( PGPKeyDBObj const *obj );
PGPBoolean				pgpKeyDBObjIsDummy( PGPKeyDBObj const *obj );
PGPBoolean				pgpKeyDBObjIsX509Dummy( PGPKeyDBObj const *obj );
PGPBoolean				pgpKeyDBObjIsReal( PGPKeyDBObj const *obj );
PGPBoolean				pgpKeyDBObjIsValid( PGPKeyDBObj const * obj);
PGPBoolean				pgpKeyIsValid( PGPKeyDBObj const * key);
PGPBoolean				pgpSubKeyIsValid(PGPKeyDBObj const * subKey);
PGPBoolean				pgpUserIDIsValid(PGPKeyDBObj const * userID);
PGPBoolean				pgpSigIsValid(PGPKeyDBObj const * cert);
PGPError				pgpKeyDBObjRefresh( PGPKeyDBObj *obj,
							PGPBoolean recurse );
PGPError				pgpKeyRefreshFromList( PGPKeyDB *kdb, PGPUInt32 *list,
							PGPSize listsize );
PGPKeyInfo *			pgpKeyToKeyInfo( PGPKeyDBObj const *obj );
PGPUserIDInfo *			pgpUserIDToUserIDInfo( PGPKeyDBObj const *obj );
PGPSigInfo *			pgpSigToSigInfo( PGPKeyDBObj const *obj );
PGPCRLInfo *			pgpCRLToCRLInfo( PGPKeyDBObj const *obj );
PGPUnkInfo *			pgpUnkToUnkInfo( PGPKeyDBObj const *obj );


/* pgpKeySet.c */

PGPBoolean				pgpKeySetIsMember(PGPKeyDBObj *obj, PGPKeySet *set);
PGPKeySet *				pgpRootSet(PGPKeyDB * db);
PGPError				pgpNewKeySetInternal(PGPKeyDBRef db,
							PGPKeySetRef * newSet);
PGPKeySet *				pgpKeyDBRootSet(PGPKeyDB * db);
PGPBoolean				pgpKeySetIsEmpty(PGPKeySetRef keys);
PGPKeyDBObj *			pgpFirstKeyInKeySet( PGPKeySet const * keySet );
PGPBoolean				pgpKeySetIsValid( PGPKeySet const *	keySet);
PGPError				pgpFreeKeySet(PGPKeySet *keys, PGPBoolean bForce);
PGPError				pgpCopyKeys_internal( PGPKeySet *keysToAdd,
							PGPKeyDB *destdb, PGPKeySet **destKeySet );
PGPError				pgpKeySetFlatten( PGPKeySet const *set,
							PGPUInt32 **keylist, PGPSize *keylistsize );
PGPError				pgpKeySetFlattenFree( PGPKeySet *set,
							PGPUInt32 **keylist, PGPSize *keylistsize );
PGPError				pgpKeySetUnflatten( PGPKeyDB *db, PGPUInt32 *keylist,
							PGPSize keylistsize, PGPKeySetRef *set);
PGPError				pgpKeySetUnflattenFree( PGPKeyDB *db,
							PGPUInt32 *keylist, PGPSize keylistsize,
							PGPKeySetRef *set);
PGPBoolean				pgpKeySetIsFlattenable( PGPKeySet const *set );
PGPError				pgpKeySetRefreshFree( PGPKeySet *set );


/* pgpKeySig.c */

PGPError				pgpCheckKeyRingSigs_internal(PGPKeySetRef keysToCheck,
							PGPKeyDBRef otherDB, PGPBoolean checkAll,
							PGPEventHandlerProcPtr eventHandler,
							PGPUserValue eventHandlerData,
							PGPKeySetRef changeSet );
PGPError				pgpPrepareToCheckKeyRingSigs_internal(
							PGPKeySetRef keysToCheck, PGPKeyDBRef otherDB,
							PGPBoolean checkAll, PGPUInt32 *nsigs,
							PGPKeySetRef changedSet );
PGPUInt32				pgpCountCheckableKeySigs(PGPKeySet *keyset,
							PGPKeyDB *signdb, PGPBoolean allflag,
							PGPBoolean revocationonly);
PGPError				pgpCheckSig_internal( PGPKeyDBRef keydb,
							PGPKeyDBObj *sig, PGPKeyDBRef otherdb,
							PGPBoolean checkAll, PGPBoolean revocationonly,
							PGPBoolean *handled, PGPBoolean *changed,
							PGPBoolean *verified );
PGPError				pgpCheckKeySigs(PGPKeySet *keyset, PGPKeyDB *otherdb,
							PGPBoolean allflag, PGPBoolean revocationonly,
							PGPBoolean frontEnd,
							PGPError (*func)(void *, PGPKeyDBObj *, int),
							void *arg, PGPKeySetRef changeSet);
PGPError				pgpHashObj (PGPHashContext *hc, PGPKeyDBObj *obj,
							PGPBoolean hashuseridlength);
PGPError				pgpHashObjParents (PGPHashContext *hc,
							PGPKeyDBObj *obj, PGPBoolean hashuseridlength);
PGPInt32				pgpSignObj(PGPKeyDBObj *obj, PGPSigSpec *spec,
							PGPByte **sig);


/* pgpRngPars.c */

PGPError				pgpKeyParse(PGPContextRef context, PGPByte const *buf,
							PGPSize len, PGPByte *pkalg, PGPByte keyID[8],
							PGPByte fp20n[20], PGPUInt16 *keybits,
							PGPUInt32 *tstamp, PGPUInt16 *validity,
							PGPByte *v3, int secretf);
PGPByte const *			pgpKeyParseNumericData(PGPByte const *buf, PGPSize len,
							PGPSize *lenp);
PGPSize					pgpKeyParsePublicPrefix(PGPByte const *buf,
							PGPSize len);
PGPByte const *			pgpSigParseExtra(PGPByte const *buf, PGPSize len,
							PGPSize *lenp);
PGPError				pgpSigParse(PGPByte const *buf, PGPSize len,
							PGPBoolean skip, PGPByte *pkalg, PGPByte keyID[8],
							PGPUInt32 *tstamp, PGPUInt32 *validity,
							PGPByte *type, PGPByte *hashalg, PGPSize *extralen,
							PGPByte *version, PGPBoolean *exportable,
							PGPBoolean *revocable, PGPByte *trustLevel,
							PGPByte *trustValue, PGPBoolean *hasRegExp,
							PGPBoolean *isX509, PGPBoolean *primaryUID);
PGPByte const *			pgpSigFindSubpacket(PGPByte const *buf,int subpacktype,
							PGPUInt32 nth, PGPSize *plen,
							PGPBoolean *pcritical, PGPBoolean *phashed,
							PGPUInt32 *pcreation, PGPUInt32 *pmatches);
PGPByte const *			pgpSigFindNAISubSubpacket(PGPByte const *buf,
							int subpacktype, PGPUInt32 nth, PGPSize *plen,
							int *pcritical, int *phashed, PGPUInt32 *pcreation,
							PGPUInt32 *pmatches);
PGPError				pgpSigSubpacketsSpec(PGPSigSpec *spec,
							PGPByte const *buf);
PGPByte const *			pgpAttrSubpacket(PGPByte const *buf, PGPSize len,
							PGPUInt32 nth, PGPUInt32 *subpacktype,
							PGPSize *plen);
PGPUInt32				pgpAttrCountSubpackets(PGPByte const *buf,
							PGPSize len);
PGPError				pgpKeyParseFingerprint16(PGPContextRef context,
							PGPByte const *kbuf, PGPSize klen,
							PGPByte *fingerprint);
PGPSize					pgpKeyBufferLength(PGPKeySpec const *ks,
							PGPByte pkalg);
PGPError				pgpKeyToBuffer(PGPByte *buf, PGPKeySpec const *ks,
							PGPByte pkalg);
PGPError				pgpCRLParse( PGPContextRef context,
							PGPByte const *buf, PGPSize len, PGPByte *version,
							PGPByte *type, PGPUInt32 *tstamp,
							PGPUInt32 *tstampnext, PGPByte const **dpoint,
							PGPSize *dpointlen );
PGPByte const *			pgpCRLFindDPoint( PGPByte const *buf, PGPSize buflen,
							PGPSize *len );
PGPByte const *			pgpCRLFindCRL( PGPByte const *buf, PGPSize buflen,
							PGPSize *len );


/* pgpRngPkt.c */

PGPSize					pgpPktBufferHeaderLen( PGPByte const *buf );
PGPSize					pgpPktByteGet(PGPFile *f, PGPSize *lenp,
							PGPSize *len1p, PGPFileOffset *posp);
PGPSize					pgpPktBodyRead(void *ptr, PGPSize len, PGPSize len1,
							PGPFile *f);
PGPUInt32				pgpPktHeaderLen (PGPByte pktbyte, PGPSize len);
PGPByte					pgpPktHeaderPut( PGPByte *buf, PGPByte pktbyte,
							PGPSize len );


/* pgpTrust.c */

PGPByte					pgpenvMarginalTrust( PGPEnv const *env );
PGPByte					pgpenvCompleteTrust( PGPEnv const *env );
PGPByte					pgpTrustToExtern(PGPUInt16 trust);
PGPUInt16				pgpTrustToIntern(PGPByte trust);
PGPByte					pgpTrustOldToExtern(PGPEnv const *env,
							PGPByte oldTrust);
PGPByte					pgpTrustExternToOld(PGPEnv const *env,
							PGPByte externTrust);
PGPByte					pgpTrustOldKeyToUserID(PGPByte keyTrust);
double					pgpTrustToDouble(PGPUInt16 t);
PGPUInt16				pgpDoubleToTrust(double d);


/* pgpTrustProp.c */

PGPUInt16				pgpKeyCalcTrust(PGPKeyDBObj *key);
PGPError				pgpPropagateTrustInternal(PGPKeySet *set,
							PGPKeyDB *altdb, PGPUInt32 const timenow,
							PGPKeySetRef changedSet );


/* pgpUnionSet.c */

PGPError				pgpGenericUnionOfKeySets(PGPKeySetRef set1,
							PGPKeySetRef set2, PGPKeySetRef * newSet);

/* pgpP11Key.c */
PGPError				pgpUnloadTCL();

/************************* Debugging Support **************************/

#define PGPValidateKeyDB( keyDB ) \
			PGPValidateParam( pgpKeyDBIsValid( keyDB ) )
#define PGPValidateKeySet( keySet ) \
			PGPValidateParam( pgpKeySetIsValid( keySet ) )
#define PGPValidateKeyList( keyList ) \
			PGPValidateParam( pgpKeyListIsValid( keyList ) )
#define PGPValidateKeyIter( keyIter ) \
			PGPValidateParam( pgpKeyIterIsValid( keyIter ) )
#define PGPValidateKey( key ) \
			PGPValidateParam( pgpKeyIsValid( key ) )
#define PGPValidateKeyOrSubKey( key ) \
			PGPValidateParam( pgpKeyIsValid( key ) || pgpSubKeyIsValid( key ) )
#define PGPValidateSubKey( subKey ) \
			PGPValidateParam( pgpSubKeyIsValid( subKey ) )
#define PGPValidateUserID( userID ) \
			PGPValidateParam( pgpUserIDIsValid( userID ) )
#define PGPValidateSig( sig ) \
			PGPValidateParam( pgpSigIsValid( sig ) )
#define PGPValidateKeyDBObj( obj ) \
			PGPValidateParam( pgpKeyDBObjIsValid( obj ) )
#define PGPValidateKeyID( id ) \
			PGPValidateParam( pgpKeyIDIsValid( id ) )

#if PGP_DEBUG	/* [ */

#define pgpaPGPKeyDBValid(db)												\
			pgpaCall(pgpaInternalPGPKeyDBValid,( pgpaCallPrefix, db, #db))
#define pgpaPGPKeySetValid(set)												\
			pgpaCall(pgpaInternalPGPKeySetValid,( pgpaCallPrefix, set, #set))
#define pgpaPGPKeyListValid(list)											\
			pgpaCall(pgpaInternalPGPKeyListValid,( pgpaCallPrefix,list,#list))
#define pgpaPGPKeyIterValid(iter )											\
			pgpaCall(pgpaInternalPGPKeyIterValid,(pgpaCallPrefix,iter,#iter ))
#define pgpaPGPKeyDBObjValid(obj )											\
			pgpaCall(pgpaInternalPGPKeyDBObjValid,(pgpaCallPrefix, obj, #obj ))
#define pgpaPGPKeyValid(key )												\
			pgpaCall(pgpaInternalPGPKeyValid,( pgpaCallPrefix, key, #key ))
#define pgpaPGPSubKeyValid(subKey )											\
			pgpaCall(pgpaInternalPGPSubKeyValid,							\
				(pgpaCallPrefix, subKey, #subKey ))
#define pgpaPGPUserIDValid(userID)											\
			pgpaCall(pgpaInternalPGPUserIDValid,							\
				(pgpaCallPrefix, userID, #userID))
#define pgpaPGPCertValid(cert)												\
			pgpaCall(pgpaInternalPGPCertValid,( pgpaCallPrefix, cert, #cert))


PGPBoolean		pgpaInternalPGPKeyDBValid(pgpaCallPrefixDef,
							PGPKeyDB const *keyDB, char const *varName);
PGPBoolean		pgpaInternalPGPKeySetValid(pgpaCallPrefixDef,
							PGPKeySet const *keySet, char const *varName);
PGPBoolean		pgpaInternalPGPKeyListValid(pgpaCallPrefixDef,
							PGPKeyList const *keyList, char const *varName);
PGPBoolean		pgpaInternalPGPKeyIterValid(pgpaCallPrefixDef,
							PGPKeyIter const *keyIter, char const *varName);
PGPBoolean		pgpaInternalPGPKeyValid(pgpaCallPrefixDef,
							PGPKeyDBObj const *key, char const *varName);
PGPBoolean		pgpaInternalPGPSubKeyValid(pgpaCallPrefixDef,
							PGPKeyDBObj const *subKey, char const *varName);
PGPBoolean		pgpaInternalPGPUserIDValid(pgpaCallPrefixDef,
							PGPKeyDBObj const *userID, char const *varName);
PGPBoolean		pgpaInternalPGPCertValid(pgpaCallPrefixDef,
							PGPKeyDBObj const *cert, char const *varName);
PGPBoolean		pgpaInternalPGPKeyDBObjValid(pgpaCallPrefixDef,
							PGPKeyDBObj const *obj, char const *varName);

#endif /* ] PGP_DEBUG */






PGP_END_C_DECLARATIONS



#endif /* Included_pgpKeyPriv_h */

/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
